package ru.yandex.direct.ytwrapper.model;

import java.util.Collection;
import java.util.Collections;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.inside.yt.kosher.cypress.YPath;

/**
 * Класс для описания YT таблицы
 */
@ParametersAreNonnullByDefault
public class YtTable {
    private final String path;

    /**
     * @param path путь до YT таблицы
     */
    public YtTable(String path) {
        this.path = YtPathUtil.generatePath(path);
    }

    /**
     * Конструирует {@link YPath} до YT таблицы.
     *
     * @return {@link YPath} до YT таблицы
     */
    public YPath ypath() {
        return ypath(Collections.emptyList());
    }

    /**
     * Конструирует {@link YPath} до YT таблицы.
     *
     * @param ytFields коллекция полей, которые нужно включить в путь
     * @return {@link YPath} до YT таблицы
     */
    public YPath ypath(Collection<YtField> ytFields) {
        YPath p = YPath.simple(path);
        if (!ytFields.isEmpty()) {
            String[] colNames = ytFields.stream().map(YtField::getName).toArray(String[]::new);
            p = p.withColumns(colNames);
        }
        return p;
    }

    /**
     * @return строка с абсолютным путем до таблицы (напр. //home/direct/db/phrases)
     */
    public String getPath() {
        return path;
    }

    /**
     * @return строка с именем таблицы (напр. phrases)
     */
    public String getName() {
        return ypath().name();
    }

    @Override
    public String toString() {
        return String.format("%s(\"%s\")", this.getClass().getSimpleName(), getPath());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YtTable table = (YtTable) o;
        return Objects.equals(path, table.path);
    }

    @Override
    public int hashCode() {
        return Objects.hash(path);
    }
}
