package ru.yandex.direct.ytwrapper.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Класс для описания ряда данных YT таблицы
 */
@ParametersAreNonnullByDefault
public class YtTableRow {
    /**
     * Специальное поле YT строк, в которое записывается значение table_index
     */
    public static final YtField<Integer> TI_FIELD = new YtField<>("ti", Integer.class);
    /**
     * Название атрибута, в котором находится индекс таблицы
     */
    public static final String TABLE_INDEX_ATTR_NAME = "table_index";

    private final List<YtField> fields;

    private Integer tableIndex;
    private YTreeMapNode data;

    public YtTableRow() {
        this(Collections.emptyList());
    }

    /**
     * @param fields список полей в ряде данных таблицы
     */
    public YtTableRow(List<YtField> fields) {
        this.fields = new ArrayList<>(fields);
    }

    public List<YtField> getFields() {
        return fields;
    }

    /**
     * Скопировать (по ссылке, без копирования значений) данные из одного ряда в другой
     */
    public void setDataFrom(YtTableRow row) {
        this.data = row.getData();
        this.tableIndex = row.getTableIndex();
    }

    /**
     * Установить данные ряда.
     * <p>
     * Если при этом в данные есть поле TI_FIELD, его значение берется, как значение индекса таблицы, иначе производится
     * проверка аттрибута {@value TABLE_INDEX_ATTR_NAME}.
     */
    void setData(YTreeMapNode data) {
        this.data = data;

        tableIndex = valueOf(TI_FIELD);
        if (tableIndex == null) {
            tableIndex = this.data.getAttribute(TABLE_INDEX_ATTR_NAME).map(YTreeNode::intValue).orElse(null);
        }
    }

    /**
     * Аналог setData, возвращающий текущий объект
     */
    YtTableRow withData(YTreeMapNode data) {
        setData(data);
        return this;
    }

    /**
     * Получить данные ряда, если данных нет, создать их и вернуть.
     */
    public YTreeMapNode getData() {
        if (data == null) {
            data = YTree.mapBuilder().buildMap();
        }
        return data;
    }

    /**
     * Получить структуру с данными только из тех полей ряда, которые указаны в переданной коллекции
     * <p>
     * Дополнительно, если задан индекс таблицы, в каждую структуру с данными добавляется поле TI_FIELD с этим индексом
     */
    YTreeMapNode getStrippedData(Collection<YtField> fieldsToStrip) {
        if (fieldsToStrip.isEmpty()) {
            fieldsToStrip = fields;
        }

        YTreeMapNode node = onlyFields(getData(), fieldsToStrip);
        if (tableIndex != null) {
            TI_FIELD.insertValue(node, tableIndex);
        }
        return node;
    }

    /**
     * Принимает YT строку {@code node} и возвращает ее копию, в которой оставлены только поля, указанные в {@code fields}
     */
    private static YTreeMapNode onlyFields(YTreeMapNode node, Collection<YtField> fields) {
        YTreeMapNode ret = YTree.mapBuilder().buildMap();
        for (YtField field : fields) {
            Optional<YTreeNode> valOption = node.get(field.getName());
            valOption.ifPresent(yTreeNode -> ret.put(field.getName(), yTreeNode));
        }
        return ret;
    }

    public Integer getTableIndex() {
        return tableIndex;
    }

    public void setTableIndex(int tableIndex) {
        this.tableIndex = tableIndex;
    }

    /**
     * Получить значение поля из ряда данных, возвращая null, если данных нет или значение поля не задано
     *
     * @param field поле
     * @param <T>   тип поля
     */
    public <T> T valueOf(YtField<T> field) {
        return valueOf(field, null);
    }

    /**
     * Получить значение поля из ряда данных, возвращая переданное значение по умолчанию, если данных нет или
     * значение поля не задано
     *
     * @param field        поле
     * @param <T>          тип поля
     * @param defaultValue значение по умолчанию
     */
    public <T> T valueOf(YtField<T> field, @Nullable T defaultValue) {
        return data != null ? field.extractValue(data, defaultValue) : defaultValue;
    }

    /**
     * Установить значение для поля. Если в ряде нет структуры с данными, она будет создана
     *
     * @param field поле
     * @param value значения для установки
     * @param <T>   тип поля
     */
    public <T> void setValue(YtField<T> field, T value) {
        field.insertValue(getData(), value);
    }

    @Override
    public String toString() {
        return String.format("%s(fields=%s, index=%s)", this.getClass().getSimpleName(), fields, tableIndex);
    }

    public List<Map<String, String>> getSchema() {
        return mapList(fields, YtField::getSchema);
    }
}
