package ru.yandex.direct.ytwrapper.schemagen;

import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.lang.model.element.Modifier;

import com.google.common.collect.ImmutableMap;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.apache.commons.lang3.text.WordUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.ytwrapper.model.YtField;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.ytwrapper.YtUtils.SCHEMA_ATTR;

/**
 * Генератор классов со схемой YT-таблиц
 * <p>
 * Пример параметров запуска: -c direct/libs/yt-wrapper/src/main/resources/schemagen.conf -p hahn.yt.yandex.net -o direct/libs/yt-wrapper/src/main/java -u
 * <p>
 * Считает, что YT-токен лежит в файле ~/.yt/token, положение токена можно перезадать с помощью ключа -t
 */
@ParametersAreNonnullByDefault
public class SchemaGenerator {
    private static final String USER_SCHEMA_ATTR = "_read_schema";
    private static final String NAME = "name";
    private static final String TYPE = "type";

    private static final String CLASS_PREFIX = "Yt";
    private static final String ROW_SUFFIX = "Row";
    private static final String TABLES_CONTAINER_NAME = "DbTables";
    private static final String PATH_PARAM = "path";
    private static final String COLS_PARAM = "cols";
    private static final String DEFAULT_PATH_FIELD = "DEFAULT_PATH";

    private static final Map<String, Class<?>> YT_FIELD_TYPE_MAPPING = ImmutableMap.<String, Class<?>>builder()
            .put("utf8", String.class)
            .put("string", String.class)
            .put("int32", Integer.class)
            .put("uint32", Integer.class)
            .put("int64", Long.class)
            .put("uint64", Long.class)
            .put("boolean", Boolean.class)
            .put("any", Object.class)
            .put("double", Double.class)
            .build();

    private final Yt yt;
    private final Path outputPath;
    private final Config config;
    private final String schemaAttr;

    private SchemaGenerator(SchemaGeneratorParams params) {
        String token = FileUtils.slurp(FileUtils.expandHome(params.token)).trim();

        yt = YtUtils.http(params.proxy, token);

        outputPath = Paths.get(params.outputPath);
        config = ConfigFactory.parseFile(Paths.get(params.configPath).toFile());
        schemaAttr = params.useUserSchema ? USER_SCHEMA_ATTR : SCHEMA_ATTR;
    }

    public static void main(String[] args) throws IOException {
        SchemaGeneratorParams params = new SchemaGeneratorParams();
        ParserWithHelp.parse(SchemaGenerator.class.getCanonicalName(), args, params);

        SchemaGenerator importer = new SchemaGenerator(params);
        importer.generate();
    }

    private void generate() throws IOException {
        Map<String, TypeSpec> generatedClasses = new HashMap<>();
        String pkg = config.getString("package");

        for (Config tableConfig : config.getConfigList("tables")) {
            String name = tableConfig.getString("name");
            String path = tableConfig.getString("path");

            TypeSpec spec = generateTableSchema(yt, name, path, pkg);
            generatedClasses.put(path, spec);
        }

        TypeSpec tablesContainer = generateDbTablesContainer(generatedClasses, pkg);
        writeTypeSpec(tablesContainer, pkg);
    }

    private void writeTypeSpec(TypeSpec spec, String pkg) throws IOException {
        JavaFile javaFile = JavaFile.builder(pkg, spec)
                .indent("    ")
                .skipJavaLangImports(true)
                .build();
        javaFile.writeTo(outputPath);
    }

    private TypeSpec generateTableSchema(Yt yt, String name, String path, String pkg) throws IOException {
        YtTable table = new YtTable(path);
        YTreeNode attrs = yt.cypress().get(table.ypath(), Cf.set(schemaAttr));
        YTreeNode attr = attrs.getAttribute(schemaAttr)
                .orElseThrow(() -> new NoSuchElementException(String.format("Could not find schema attribute '%s'", schemaAttr)));

        String className = CLASS_PREFIX + name;
        TypeSpec.Builder classBuilder = TypeSpec.classBuilder(className)
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(ParametersAreNonnullByDefault.class)
                .addJavadoc("Класс с описанием YT таблицы $1L\n"
                                + "Автоматически сгенерировано {@link ru.yandex.direct.ytwrapper.schemagen.SchemaGenerator}\n",
                        path)
                .superclass(YtTable.class);

        TypeSpec.Builder rowClassBuilder = TypeSpec.classBuilder(className + ROW_SUFFIX)
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(ParametersAreNonnullByDefault.class)
                .addJavadoc("Класс с описанием ряда YT таблицы $1L\n"
                                + "Автоматически сгенерировано {@link ru.yandex.direct.ytwrapper.schemagen.SchemaGenerator}\n",
                        path)
                .superclass(YtTableRow.class);

        for (YTreeNode node : attr.asList()) {
            Map<String, YTreeNode> fieldAttr = node.asMap();
            String fieldName = fieldAttr.get(NAME).stringValue();
            String ytFieldType = fieldAttr.get(TYPE).stringValue();

            Type fieldType = YT_FIELD_TYPE_MAPPING.get(ytFieldType);
            if (fieldType == null) {
                throw new RuntimeException(
                        String.format("Unmapped field type %s of field %s.%s", ytFieldType, path, fieldName));
            }

            TypeName typeName = ParameterizedTypeName.get(YtField.class, fieldType);
            FieldSpec.Builder fieldBuilder = FieldSpec.builder(typeName,
                    fieldName.toUpperCase(),
                    Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL);
            fieldBuilder.initializer("new YtField<>($1S, $2L.class)", fieldName, fieldType.getTypeName());
            classBuilder.addField(fieldBuilder.build());

            StringBuilder builder = new StringBuilder();
            for (String s : fieldName.split("_")) {
                builder.append(WordUtils.capitalize(s));
            }

            rowClassBuilder.addMethod(MethodSpec.methodBuilder(String.format("get%s", builder.toString()))
                    .addStatement("return valueOf($1L.$2L)", className, fieldName.toUpperCase())
                    .addModifiers(Modifier.PUBLIC)
                    .returns(fieldType)
                    .build()
            );

            String paramName = WordUtils.uncapitalize(builder.toString());
            rowClassBuilder.addMethod(MethodSpec.methodBuilder(String.format("set%s", builder.toString()))
                    .addParameter(fieldType, paramName)
                    .addStatement("setValue($1L.$2L, $3L)", className, fieldName.toUpperCase(), paramName)
                    .addModifiers(Modifier.PUBLIC)
                    .build()
            );
        }

        FieldSpec.Builder defaultPathFieldBuilder = FieldSpec.builder(String.class,
                DEFAULT_PATH_FIELD,
                Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL);
        defaultPathFieldBuilder.initializer("$1S", path);
        FieldSpec defaultPathField = defaultPathFieldBuilder.build();
        classBuilder.addField(defaultPathField);

        classBuilder.addMethod(MethodSpec.constructorBuilder()
                .addStatement("super($1L)", defaultPathField.name)
                .addModifiers(Modifier.PUBLIC)
                .build()
        );

        classBuilder.addMethod(MethodSpec.constructorBuilder()
                .addParameter(String.class, PATH_PARAM)
                .addStatement("super($1L)", PATH_PARAM)
                .addModifiers(Modifier.PUBLIC)
                .build()
        );

        rowClassBuilder.addMethod(MethodSpec.constructorBuilder()
                .addStatement("super()")
                .addModifiers(Modifier.PUBLIC)
                .build());

        rowClassBuilder.addMethod(MethodSpec.constructorBuilder()
                .addParameter(ParameterizedTypeName.get(List.class, YtField.class), COLS_PARAM)
                .addStatement("super($1L)", COLS_PARAM)
                .addModifiers(Modifier.PUBLIC)
                .build());

        TypeSpec spec = classBuilder.build();
        writeTypeSpec(spec, pkg);
        writeTypeSpec(rowClassBuilder.build(), pkg);
        return spec;
    }

    private TypeSpec generateDbTablesContainer(Map<String, TypeSpec> tableClasses, String pkg) {
        TypeSpec.Builder classBuilder = TypeSpec.classBuilder(CLASS_PREFIX + TABLES_CONTAINER_NAME)
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc("Инстансы объектов описывающих таблицы\n"
                        + "Автоматически сгенерировано {@link ru.yandex.direct.ytwrapper.schemagen.SchemaGenerator}\n");

        for (Map.Entry<String, TypeSpec> entry : tableClasses.entrySet()) {
            TypeSpec tableClass = entry.getValue();
            String paramName = tableClass.name.toUpperCase().replaceFirst("^" + CLASS_PREFIX.toUpperCase(), "");
            FieldSpec.Builder fieldBuilder = FieldSpec
                    .builder(ClassName.get(pkg, tableClass.name),
                            paramName,
                            Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL);
            fieldBuilder.initializer("new $1L()", tableClass.name);
            classBuilder.addField(fieldBuilder.build());
        }

        classBuilder.addMethod(MethodSpec.constructorBuilder().addModifiers(Modifier.PRIVATE).build());

        return classBuilder.build();
    }
}
