package ru.yandex.direct.ytwrapper.specs;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.exceptions.SpecGenerationException;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableRow;

@ParametersAreNonnullByDefault
public abstract class AppendableSpecBuilder {
    private static final Logger logger = LoggerFactory.getLogger(AppendableSpecBuilder.class);

    public static class TableRowPair {
        private final YtTable table;
        private final YtTableRow row;

        public TableRowPair(YtTable table, YtTableRow row) {
            this.table = table;
            this.row = row;
        }

        public YtTable getTable() {
            return table;
        }

        public YtTableRow getRow() {
            return row;
        }

        @Override
        public String toString() {
            return table.toString();
        }
    }

    private List<TableRowPair> inputTables = new ArrayList<>();
    private List<YtTable> outputTables = new ArrayList<>();
    private Duration operationTimeout;

    private AppendableSpecBuilder nextOperation;

    public Duration getOperationTimeout() {
        return operationTimeout;
    }

    public AppendableSpecBuilder setOperationTimeout(Duration operationTimeout) {
        this.operationTimeout = operationTimeout;
        return this;
    }

    public AppendableSpecBuilder setInputTable(YtTable ytTable, YtTableRow ytTableRow) {
        if (!inputTables.isEmpty()) {
            throw new SpecGenerationException("Cannot set input table because it already presents");
        }
        inputTables.add(pair(ytTable, ytTableRow));
        return this;
    }

    public AppendableSpecBuilder setInputTable(YtTable ytTable) {
        return setInputTable(ytTable, new YtTableRow());
    }

    public AppendableSpecBuilder addInputTables(Collection<YtTable> ytTables) {
        ytTables.forEach(ytTable -> addInputTable(ytTable, new YtTableRow()));
        return this;
    }

    public AppendableSpecBuilder addInputTable(YtTable ytTable) {
        return addInputTable(ytTable, new YtTableRow());
    }

    public AppendableSpecBuilder addInputTable(TableRowPair tableRowPair) {
        inputTables.add(tableRowPair);
        return this;
    }

    public AppendableSpecBuilder addInputTablesAndRows(Collection<TableRowPair> tableRowPairs) {
        inputTables.addAll(tableRowPairs);
        return this;
    }

    public AppendableSpecBuilder addInputTable(YtTable ytTable, YtTableRow ytTableRow) {
        inputTables.add(pair(ytTable, ytTableRow));
        return this;
    }

    public List<TableRowPair> getInputTables() {
        return inputTables;
    }

    public AppendableSpecBuilder addOutputTable(YtTable ytTable) {
        outputTables.add(ytTable);
        return this;
    }

    public List<YtTable> getOutputTables() {
        return outputTables;
    }

    public void appendTo(AppendableSpecBuilder builder) {
        builder.append(this);
    }

    public AppendableSpecBuilder append(AppendableSpecBuilder appendableSpecBuilder) {
        if (nextOperation != null) {
            nextOperation.append(appendableSpecBuilder);
        } else {
            nextOperation = appendableSpecBuilder;
        }
        return this;
    }

    public abstract void validateCurrent();

    public OperationSpec build() {
        if (nextOperation == null) {
            return buildCurrent();
        }

        List<OperationSpec> operationSpecs = new ArrayList<>();
        List<YtTable> needDelete = new ArrayList<>();

        AppendableSpecBuilder builder = this;
        while (builder != null) {
            if (builder.nextOperation != null && builder.outputTables.isEmpty()) {
                YtTable temporaryTable = new YtTable(YtPathUtil.generateTemporaryPath());
                builder.outputTables.add(temporaryTable);
                builder.nextOperation.setInputTable(temporaryTable);
                needDelete.add(temporaryTable);
            }
            validateCurrent();

            logger.info("Building {}", builder);
            operationSpecs.add(builder.buildCurrent());

            builder = builder.nextOperation;
        }

        for (YtTable ytTable : needDelete) {
            AppendableSpecBuilder specBuilder = new DeleteTableSpecBuilder().addInputTable(ytTable);
            logger.info("Adding {}", specBuilder);
            operationSpecs.add(specBuilder.buildCurrent());
        }

        return new TransactionalOperationSpec(operationSpecs);
    }

    protected abstract OperationSpec buildCurrent();

    public static TableRowPair pair(YtTable table, YtTableRow row) {
        return new TableRowPair(table, row);
    }
}
