package ru.yandex.direct.ytwrapper.specs;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.exceptions.SpecGenerationException;
import ru.yandex.direct.ytwrapper.model.YtField;
import ru.yandex.direct.ytwrapper.model.YtMapper;
import ru.yandex.direct.ytwrapper.model.YtReducer;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.operations.specs.MapReduceSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapperOrReducerSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapperSpec;
import ru.yandex.inside.yt.kosher.operations.specs.ReducerSpec;
import ru.yandex.misc.dataSize.DataSize;

@ParametersAreNonnullByDefault
public class MapReduceSpecBuilder extends AppendableSpecBuilder {
    private static final Duration DEFAULT_TIMEOUT = Duration.ofHours(4);

    private Integer mapJobCount;
    private YtMapper mapper;
    private DataSize mapperMemoryLimit;
    private List<YtField<?>> reduceByFields = new ArrayList<>();
    private List<YtField<?>> sortByFields = new ArrayList<>();
    private YtReducer reducer;
    private DataSize reducerMemoryLimit;
    private Integer partitionCount;
    private Integer partitionJobCount;
    private DataSize maxDataSizePerJob;
    private String pool;

    public MapReduceSpecBuilder addInputTables(List<TableRowPair> ytTables) {
        ytTables.forEach(this::addInputTable);
        return this;
    }

    public MapReduceSpecBuilder setMapJobCount(Integer mapJobCount) {
        this.mapJobCount = mapJobCount;
        return this;
    }

    public MapReduceSpecBuilder setMapper(YtMapper mapper) {
        this.mapper = mapper;
        return this;
    }

    public MapReduceSpecBuilder setMapperMemoryLimit(DataSize memoryLimit) {
        this.mapperMemoryLimit = memoryLimit;
        return this;
    }

    public MapReduceSpecBuilder setReducerMemoryLimit(DataSize memoryLimit) {
        this.reducerMemoryLimit = memoryLimit;
        return this;
    }

    public MapReduceSpecBuilder addReduceByField(YtField<?> ytField) {
        reduceByFields.add(ytField);
        return this;
    }

    public MapReduceSpecBuilder addSortByField(YtField<?> ytField) {
        sortByFields.add(ytField);
        return this;
    }

    public MapReduceSpecBuilder setReducer(YtReducer reducer) {
        this.reducer = reducer;
        return this;
    }

    public MapReduceSpecBuilder setPartitionCount(int partitionCount) {
        this.partitionCount = partitionCount;
        return this;
    }

    public MapReduceSpecBuilder setPartitionJobCount(int partitionJobCount) {
        this.partitionJobCount = partitionJobCount;
        return this;
    }

    public MapReduceSpecBuilder setMaxDataSizePerJob(DataSize dataSizePerJob) {
        this.maxDataSizePerJob = dataSizePerJob;
        return this;
    }

    public MapReduceSpecBuilder setPool(String pool) {
        this.pool = pool;
        return this;
    }

    @Override
    public void validateCurrent() {
        if (getInputTables().isEmpty()) {
            throw new SpecGenerationException("No input tables provided");
        } else if (getOutputTables().isEmpty()) {
            throw new SpecGenerationException("No output table provided");
        } else if (reduceByFields.isEmpty()) {
            throw new SpecGenerationException("No reduce-by fields provided");
        } else if (reducer == null) {
            throw new SpecGenerationException("No reducer class provided");
        } else if (mapper == null) {
            throw new SpecGenerationException("No mapper class provided");
        }
        if (!sortByFields.isEmpty()) {
            for (int i = 0; i < reduceByFields.size(); i++) {
                if (i >= sortByFields.size()) {
                    throw new SpecGenerationException("reduce-by fields list cannot be longer than sort-by list");
                } else if (!reduceByFields.get(i).equals(sortByFields.get(i))) {
                    throw new SpecGenerationException("sort-by fields list should start with full reduce-by list");
                }
            }
        }
    }

    @Override
    protected OperationSpec buildCurrent() {
        MapperSpec.Builder mapperSpecBuilder = MapperSpec.builder()
                .setJavaOptions(MapperOrReducerSpec.DEFAULT_JAVA_OPTIONS
                        .withOption("-Djava.library.path=./"))
                .setMapper(mapper)
                .setOutputTables(1);
        if (mapperMemoryLimit != null) {
            mapperSpecBuilder.setMemoryLimit(ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(mapperMemoryLimit.toBytes()));
        }

        ReducerSpec.Builder reducerSpecBuilder = ReducerSpec.builder()
                .setJavaOptions(MapperOrReducerSpec.DEFAULT_JAVA_OPTIONS
                        .withOption("-Djava.library.path=./"))
                .setReducer(reducer)
                .setOutputTables(getOutputTables().size());
        if (reducerMemoryLimit != null) {
            reducerSpecBuilder.setMemoryLimit(ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(reducerMemoryLimit.toBytes()));
        }

        MapReduceSpec.Builder mrSpecBuilder = MapReduceSpec.builder()
                .setInputTables(Cf.wrap(getInputTables()).map(it -> it.getTable().ypath(it.getRow().getFields())))
                .setOutputTables(Cf.wrap(getOutputTables()).map(YtTable::ypath))
                .setMapperSpec(mapperSpecBuilder.build())
                .setSortBy(sortByFields.isEmpty() ? Cf.wrap(reduceByFields).map(YtField::getName)
                        : Cf.wrap(sortByFields).map(YtField::getName))
                .setReduceBy(Cf.wrap(reduceByFields).map(YtField::getName))
                .setReducerSpec(reducerSpecBuilder.build());

        if (partitionCount != null) {
            mrSpecBuilder.setPartitionCount(partitionCount);
        }
        if (partitionJobCount != null) {
            mrSpecBuilder.setPartitionJobCount(partitionJobCount);
        }
        if (maxDataSizePerJob != null) {
            mrSpecBuilder.setMaxDataSizePerSortJob(ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(maxDataSizePerJob.toBytes()));
        }
        if (mapJobCount != null) {
            mrSpecBuilder.setMapJobCount(mapJobCount);
        }
        if (pool != null) {
            mrSpecBuilder.setPool(pool);
        }

        MapReduceSpec mrSpec = mrSpecBuilder.build();

        return new SingleOperationSpec(mrSpec, getOperationTimeout() == null ? DEFAULT_TIMEOUT : getOperationTimeout());
    }

    @Override
    public String toString() {
        return String.format("MapReduce(input=%s, output=%s, mapper=%s, sortBy=%s, reduceBy=%s, reducer=%s)",
                getInputTables(), getOutputTables(), mapper.getClass(),
                sortByFields, reduceByFields, reducer.getClass());
    }
}
