package ru.yandex.direct.ytwrapper.specs;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.exceptions.SpecGenerationException;
import ru.yandex.direct.ytwrapper.model.YtField;
import ru.yandex.direct.ytwrapper.model.YtReducer;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.operations.specs.MapperOrReducerSpec;
import ru.yandex.inside.yt.kosher.operations.specs.ReduceSpec;
import ru.yandex.inside.yt.kosher.operations.specs.ReducerSpec;
import ru.yandex.misc.dataSize.DataSize;

@ParametersAreNonnullByDefault
public class ReduceSpecBuilder extends AppendableSpecBuilder {
    private static final Duration DEFAULT_TIMEOUT = Duration.ofHours(4);

    private List<YtField<?>> reduceByFields = new ArrayList<>();
    private YtReducer reducer;
    private DataSize reducerMemoryLimit;
    private DataSize reducerXmx;
    private Integer partitionCount;
    private Integer partitionJobCount;
    private DataSize dataSizePerJob;
    private Boolean reduceUseTmpFs;
    private DataSize reducerTmpFsSize;
    private String pool;

    public ReduceSpecBuilder addInputTables(List<TableRowPair> ytTables) {
        ytTables.forEach(this::addInputTable);
        return this;
    }

    public ReduceSpecBuilder setReducerMemoryLimit(DataSize memoryLimit) {
        this.reducerMemoryLimit = memoryLimit;
        return this;
    }

    public ReduceSpecBuilder setReducerXmx(DataSize reducerXmx) {
        this.reducerXmx = reducerXmx;
        return this;
    }

    public ReduceSpecBuilder addReduceByFields(Collection<YtField<?>> ytFields) {
        reduceByFields.addAll(ytFields);
        return this;
    }

    public ReduceSpecBuilder addReduceByField(YtField<?> ytField) {
        reduceByFields.add(ytField);
        return this;
    }

    public ReduceSpecBuilder setReducer(YtReducer reducer) {
        this.reducer = reducer;
        return this;
    }

    public ReduceSpecBuilder setPartitionCount(int partitionCount) {
        this.partitionCount = partitionCount;
        return this;
    }

    public ReduceSpecBuilder setPartitionJobCount(int partitionJobCount) {
        this.partitionJobCount = partitionJobCount;
        return this;
    }

    public ReduceSpecBuilder setDataSizePerJob(DataSize dataSizePerJob) {
        this.dataSizePerJob = dataSizePerJob;
        return this;
    }

    public ReduceSpecBuilder setReducerUseTmpFs(boolean useTmpFs) {
        this.reduceUseTmpFs = useTmpFs;
        return this;
    }

    public ReduceSpecBuilder setReducerTmpFsSize(DataSize reducerTmpFsSize) {
        this.reducerTmpFsSize = reducerTmpFsSize;
        return this;
    }

    public ReduceSpecBuilder setPool(String pool) {
        this.pool = pool;
        return this;
    }

    @Override
    public void validateCurrent() {
        if (getInputTables().isEmpty()) {
            throw new SpecGenerationException("No input tables provided");
        } else if (getOutputTables().isEmpty()) {
            throw new SpecGenerationException("No output table provided");
        } else if (reduceByFields.isEmpty()) {
            throw new SpecGenerationException("No reduce-by fields provided");
        } else if (reducer == null) {
            throw new SpecGenerationException("No reducer class provided");
        }
    }

    @Override
    protected OperationSpec buildCurrent() {
        ReducerSpec.Builder reducerSpecBuilder = ReducerSpec.builder()
                .setJavaOptions(MapperOrReducerSpec.DEFAULT_JAVA_OPTIONS
                        .withXmx(reducerXmx != null ? ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(reducerXmx.toBytes()) : MapperOrReducerSpec.DEFAULT_MEMORY_LIMIT)
                        .withOption("-Djava.library.path=./"))
                .setReducer(reducer)
                .setOutputTables(getOutputTables().size());
        if (reducerMemoryLimit != null) {
            reducerSpecBuilder = reducerSpecBuilder.setMemoryLimit(ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(reducerMemoryLimit.toBytes()));
        }
        if (reduceUseTmpFs != null) {
            reducerSpecBuilder = reducerSpecBuilder
                    .setUseTmpfs(reduceUseTmpFs)
                    .setTmpfsSize(ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(reducerTmpFsSize.toBytes()));
        }

        ReduceSpec.Builder reduceSpecBuilder = ReduceSpec.builder()
                .setInputTables(Cf.wrap(getInputTables()).map(it -> it.getTable().ypath(it.getRow().getFields())))
                .setOutputTables(Cf.wrap(getOutputTables()).map(YtTable::ypath))
                .setReduceBy(Cf.wrap(reduceByFields).map(YtField::getName))
                .setReducerSpec(reducerSpecBuilder.build());
        if (dataSizePerJob != null) {
            reduceSpecBuilder.setDataSizePerJob(ru.yandex.inside.yt.kosher.common.DataSize.fromBytes(dataSizePerJob.toBytes()));
        }

        ReduceSpec reduceSpec = reduceSpecBuilder.build();

        if (pool != null) {
            reduceSpec.getPool();
        }
        return new SingleOperationSpec(reduceSpec,
                getOperationTimeout() == null ? DEFAULT_TIMEOUT : getOperationTimeout());
    }

    @Override
    public String toString() {
        return String.format("Reduce(input=%s, output=%s, reduceBy=%s, reducer=%s)",
                getInputTables(), getOutputTables(), reduceByFields, reducer.getClass());
    }
}
