package ru.yandex.direct.ytwrapper.specs;

import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.exceptions.SpecGenerationException;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.direct.ytwrapper.model.attributes.CompressionCodecAttr;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.operations.specs.MergeSpec;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.inside.yt.kosher.operations.specs.MergeMode.SORTED;

@ParametersAreNonnullByDefault
public class SetTableAttributesSpecBuilder extends AppendableSpecBuilder {
    private static final Duration DEFAULT_TIMEOUT = Duration.ofHours(1);

    private YtTableRow schema;
    private OptimizeForAttr optimizeFor;
    private CompressionCodecAttr compressionCodec;
    private Boolean external;
    private Boolean uniqueKeys;

    public SetTableAttributesSpecBuilder setOutputTable(YtTable ytTable) {
        return (SetTableAttributesSpecBuilder) addOutputTable(ytTable);
    }

    private Map<String, YTreeNode> buildAttrs() {
        ImmutableMap.Builder<String, YTreeNode> builder = ImmutableMap.builder();
        if (schema != null) {
            builder.put(YtUtils.SCHEMA_ATTR, getYTreeSchema());
        }
        if (optimizeFor != null) {
            builder.put("optimize_for", YTree.stringNode(optimizeFor.getText()));
        }
        if (compressionCodec != null) {
            builder.put("compression_codec", YTree.stringNode(compressionCodec.name().toLowerCase()));
        }
        if (external != null) {
            builder.put("external", YTree.booleanNode(external));
        }

        return builder.build();
    }

    private YTreeNode getYTreeSchema() {
        YTreeBuilder schemaBuilder = YTree.builder();
        if (uniqueKeys != null) {
            schemaBuilder.beginAttributes();
            schemaBuilder
                    .key("unique_keys")
                    .value(YTree.booleanNode(uniqueKeys));
            schemaBuilder.endAttributes();
        }
        schemaBuilder.beginList();
        for (Map<String, String> fieldSchema : schema.getSchema()) {
            YTreeBuilder fieldBuilder = YTree.mapBuilder();
            for (Map.Entry<String, String> stringStringEntry : fieldSchema.entrySet()) {
                fieldBuilder
                        .key(stringStringEntry.getKey())
                        .value(stringStringEntry.getValue());
            }
            schemaBuilder.value(fieldBuilder.buildMap());
        }
        schemaBuilder.endList();
        return schemaBuilder.build();
    }

    @Override
    protected OperationSpec buildCurrent() {
        MergeSpec spec = MergeSpec.builder()
                .setInputTables(Cf.wrap(getInputTables()).map(it -> it.getTable().ypath()))
                .setOutputTable(getOutputTables().get(0).ypath())
                .setOutputTableAttributes(Cf.wrap(buildAttrs()))
                .setMergeMode(SORTED)
                .build();
        return new SetTableAttributesOperationSpec(getOutputTables().get(0).ypath(), spec,
                getOperationTimeout() == null ? DEFAULT_TIMEOUT : getOperationTimeout());
    }

    @Override
    public void validateCurrent() {
        if (getInputTables().isEmpty()) {
            throw new SpecGenerationException("No tables to merge");
        } else if (getOutputTables().isEmpty()) {
            throw new SpecGenerationException("No source tables provided");
        } else if (getOutputTables().size() > 1) {
            throw new SpecGenerationException("Can have only one source table");
        } else if (buildAttrs().isEmpty()) {
            throw new SpecGenerationException("You should provide at least one argument");
        }

        Set<YtTable> inputTables = getInputTables().stream().map(TableRowPair::getTable).collect(Collectors.toSet());
        if (inputTables.contains(getOutputTables().get(0))) {
            throw new SpecGenerationException("You should provide at least one argument");
        }
    }

    @Override
    public String toString() {
        return String.format("MapReduce(input=%s, output=%s, attrs=%s)",
                getInputTables(), getOutputTables(), buildAttrs());
    }

    public YtTableRow getSchema() {
        return schema;
    }

    public SetTableAttributesSpecBuilder setSchema(YtTableRow schema) {
        this.schema = schema;
        return this;
    }

    public OptimizeForAttr getOptimizeFor() {
        return optimizeFor;
    }

    public SetTableAttributesSpecBuilder setOptimizeFor(OptimizeForAttr optimizeFor) {
        this.optimizeFor = optimizeFor;
        return this;
    }

    public CompressionCodecAttr getCompressionCodec() {
        return compressionCodec;
    }

    public SetTableAttributesSpecBuilder setCompressionCodec(CompressionCodecAttr compressionCodec) {
        this.compressionCodec = compressionCodec;
        return this;
    }

    public Boolean getExternal() {
        return external;
    }

    public SetTableAttributesSpecBuilder setExternal(Boolean external) {
        this.external = external;
        return this;
    }

    public Boolean getUniqueKeys() {
        return uniqueKeys;
    }

    public SetTableAttributesSpecBuilder setUniqueKeys(Boolean uniqueKeys) {
        this.uniqueKeys = uniqueKeys;
        return this;
    }
}
