package ru.yandex.direct.ytwrapper.specs;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;

/**
 * Спека, описывающая выполнение нескольких операций в одной транзакции.
 */
@ParametersAreNonnullByDefault
public class TransactionalOperationSpec implements OperationSpec {
    private static final Logger logger = LoggerFactory.getLogger(TransactionalOperationSpec.class);

    private final List<OperationSpec> operationSpecs;

    /**
     * @param spec Список операций для выполнения. Будут выполнены в том порядке, в котором они переданы
     */
    TransactionalOperationSpec(List<OperationSpec> spec) {
        operationSpecs = Collections.unmodifiableList(new ArrayList<>(spec));
    }

    public List<OperationSpec> getOperationSpecs() {
        return operationSpecs;
    }

    @Override
    public void run(Yt yt) {
        YtTransactionsUtils.withTransaction(
                yt,
                Duration.ofMinutes(1),
                Optional.of(Duration.ofSeconds(15)),
                transaction -> {
                    GUID transactionUID = transaction.getId();
                    logger.info("Run with YT-transaction {}", transactionUID);
                    try {
                        run(yt, transactionUID);
                    } finally {
                        logger.info("Closing YT-transaction {}", transactionUID);
                    }
                    return true;
                });
    }

    @Override
    public void run(Yt yt, GUID transactionUID) {
        for (OperationSpec spec : operationSpecs) {
            spec.run(yt, transactionUID);
        }
    }
}
