package ru.yandex.direct.ytwrapper.utils;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;

public final class FreshestClusterFinder {
    private static final Logger logger = LoggerFactory.getLogger(FreshestClusterFinder.class);
    private static final Duration HTTP_CLIENT_SOCKET_TIMEOUT = Duration.ofMillis(10_000);
    private static final Duration HTTP_CLIENT_CONNECT_TIMEOUT = Duration.ofMillis(10_000);
    private static final int SIMPLE_COMMANDS_RETRIES = 1;

    private FreshestClusterFinder() {
    }

    public static YtCluster getFreshestCluster(YtProvider ytProvider, YtTable ytTable, List<YtCluster> ytClusters) {
        if (ytClusters == null || ytClusters.isEmpty()) {
            return null;
        }

        var maxTime = LocalDateTime.now().minusDays(10);
        YtCluster freshestCluster = null;

        for (var ytCluster : ytClusters) {
            try {
                var ytOperator = ytProvider.getOperator(ytCluster, HTTP_CLIENT_SOCKET_TIMEOUT, HTTP_CLIENT_CONNECT_TIMEOUT, SIMPLE_COMMANDS_RETRIES);
                var modificationTime = ytOperator.readTableModificationTime(ytTable);
                var uploadTime = LocalDateTime.parse(modificationTime, DateTimeFormatter.ISO_OFFSET_DATE_TIME);

                if (uploadTime.isAfter(maxTime)) {
                    maxTime = uploadTime;
                    freshestCluster = ytCluster;
                }
            } catch (RuntimeException ex) {
                logger.warn(
                        String.format("Failed to get modificationTime for %s on cluster %s",
                                ytTable.getPath(),
                                ytCluster.getName()),
                        ex);
            }
        }

        return freshestCluster;
    }
}
