package ru.yandex.direct.zorafetcher;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.utils.InterruptedRuntimeException;

@ParametersAreNonnullByDefault
public class ZoraFetcher {

    private static final Logger logger = LoggerFactory.getLogger(ZoraFetcher.class);

    private final ParallelFetcherFactory parallelFetcherFactory;
    private final ParallelFetcherFactory goZoraParallelFetcherFactory;

    public ZoraFetcher(AsyncHttpClient asyncHttpClient,
                       AsyncHttpClient goZoraAsyncHttpClient,
                       ZoraFetcherSettings settings) {
        FetcherSettings fetcherSettings = new FetcherSettings()
                .withParallel(settings.getParallel())
                .withConnectTimeout(settings.getConnectTimeout())
                .withGlobalTimeout(settings.getGlobalTimeout())
                .withRequestTimeout(settings.getRequestTimeout())
                .withRequestRetries(settings.getRequestRetries())
                .withMetricRegistry(settings.getMetricRegistry());
        this.parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, fetcherSettings);
        this.goZoraParallelFetcherFactory = new ParallelFetcherFactory(goZoraAsyncHttpClient, fetcherSettings);
    }

    public Result<ZoraResponse> fetch(ZoraRequest request,
                                      boolean useGoZora) {
        return fetch(List.of(request), useGoZora).get(request.getId());
    }

    public Map<Long, Result<ZoraResponse>> fetch(List<ZoraRequest> requests,
                                                 boolean useGoZora) {
        logger.debug("Requests: {}", requests);
        try (
                ParallelFetcher<ZoraResponse> fetcher = useGoZora ?
                        goZoraParallelFetcherFactory.getParallelFetcher() : parallelFetcherFactory.getParallelFetcher()
        ) {
            Map<Long, Result<ZoraResponse>> result = fetcher.execute(requests);
            logger.debug("Response: {}", result);
            return result;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException();
        }
    }
}
