package ru.yandex.direct.zorafetcher;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Конфигурация для {@link ZoraFetcher}
 */
@ParametersAreNonnullByDefault
public class ZoraFetcherSettings {
    private final int parallel;
    private final Duration connectTimeout;
    private final Duration globalTimeout;
    private final Duration requestTimeout;
    private final int requestRetries;
    private final MetricRegistry metricRegistry;

    ZoraFetcherSettings(int parallel, Duration connectTimeout, Duration globalTimeout,
                        Duration requestTimeout, int requestRetries, MetricRegistry metricRegistry) {
        this.parallel = parallel;
        this.connectTimeout = connectTimeout;
        this.globalTimeout = globalTimeout;
        this.requestTimeout = requestTimeout;
        this.requestRetries = requestRetries;
        this.metricRegistry = metricRegistry;
    }

    int getParallel() {
        return parallel;
    }

    Duration getConnectTimeout() {
        return connectTimeout;
    }

    Duration getGlobalTimeout() {
        return globalTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    int getRequestRetries() {
        return requestRetries;
    }

    public MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }

    public static ZoraFetcherSettingsBuilder builder() {
        return new ZoraFetcherSettingsBuilder();
    }

    public static final class ZoraFetcherSettingsBuilder {
        private static final int DEFAULT_PARALLEL = 4;
        private static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofMillis(300);
        private static final Duration DEFAULT_GLOBAL_TIMEOUT = Duration.ofSeconds(60);
        private static final Duration DEFAULT_REQUEST_TIMEOUT = Duration.ofSeconds(5);
        private static final int DEFAULT_REQUEST_RETRIES = 0;

        private int parallel = DEFAULT_PARALLEL;
        private Duration connectTimeout = DEFAULT_CONNECT_TIMEOUT;
        private Duration globalTimeout = DEFAULT_GLOBAL_TIMEOUT;
        private Duration requestTimeout = DEFAULT_REQUEST_TIMEOUT;
        private int requestRetries = DEFAULT_REQUEST_RETRIES;
        private MetricRegistry metricRegistry;


        private ZoraFetcherSettingsBuilder() {
        }

        public ZoraFetcherSettingsBuilder withParallel(int parallel) {
            this.parallel = parallel;
            return this;
        }

        public ZoraFetcherSettingsBuilder withConnectTimeout(Duration connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public ZoraFetcherSettingsBuilder withGlobalTimeout(Duration globalTimeout) {
            this.globalTimeout = globalTimeout;
            return this;
        }

        public ZoraFetcherSettingsBuilder withRequestTimeout(Duration requestTimeout) {
            this.requestTimeout = requestTimeout;
            return this;
        }

        public ZoraFetcherSettingsBuilder withRequestRetries(int requestRetries) {
            this.requestRetries = requestRetries;
            return this;
        }

        public ZoraFetcherSettingsBuilder withMetricRegistry(MetricRegistry metricRegistry) {
            this.metricRegistry = metricRegistry;
            return this;
        }

        public ZoraFetcherSettings build() {
            return new ZoraFetcherSettings(
                    parallel, checkNotNull(connectTimeout), checkNotNull(globalTimeout), checkNotNull(requestTimeout),
                    requestRetries, metricRegistry);
        }
    }
}
