package ru.yandex.direct.zorafetcher;

import java.nio.charset.StandardCharsets;
import java.time.Duration;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.handler.codec.http.DefaultHttpHeaders;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.proxy.ProxyServer;

import static org.apache.http.HttpHeaders.ACCEPT_ENCODING;
import static org.apache.http.HttpHeaders.USER_AGENT;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class ZoraGoRequestCreator implements ZoraRequestCreator {

    private static final String X_YA_CLIENT_ID = "X-Ya-Client-Id";
    private static final String X_YA_FOLLOW_REDIRS = "X-Ya-Follow-Redirects";
    private static final String X_YA_REQ_ID = "X-Ya-Req-Id";
    private static final String X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";
    private static final String X_YA_IGNORE_CERTS = "X-Ya-Ignore-Certs";

    private static final String DEFAULT_USER_AGENT =
            "Mozilla/5.0 (compatible; YaDirectFetcher/1.0; Dyatel; +http://yandex.com/bots)";
    private static final String DEFAULT_ACCEPT_ENCODING =
            "gzip, deflate";

    private final String zoraSourceName;
    private final String zoraHost;
    private final Integer zoraPort;

    public ZoraGoRequestCreator(String zoraHost, Integer zoraPort, String zoraSourceName) {
        this.zoraSourceName = zoraSourceName;
        this.zoraHost = zoraHost;
        this.zoraPort = zoraPort;
    }

    @Override
    public Request createGetRequest(String url,
                                    String yandexTraceHeaderValue,
                                    Duration responseTimeout,
                                    String tvmTicket,
                                    boolean followRedirects,
                                    @Nullable String userAgent) {
        var proxy = new ProxyServer.Builder(zoraHost, zoraPort);
        proxy.setCustomHeaders(req -> {
            var headers = new DefaultHttpHeaders().clear();
            headers.add(X_YA_SERVICE_TICKET, tvmTicket);
            headers.add(X_YA_CLIENT_ID, zoraSourceName);
            return headers;
        });

        RequestBuilder requestBuilder = new RequestBuilder()
                .setUrl(url)
                .setHeader(X_YA_REQ_ID, yandexTraceHeaderValue)
                .setHeader(X_YA_FOLLOW_REDIRS, followRedirects ? "true" : "false")
                .setHeader(X_YA_IGNORE_CERTS, "true")
                .setHeader(USER_AGENT, nvl(userAgent, DEFAULT_USER_AGENT))
                .setHeader(ACCEPT_ENCODING, DEFAULT_ACCEPT_ENCODING)
                .setProxyServer(proxy);

        return requestBuilder.setCharset(StandardCharsets.UTF_8)
                // это timeout для запроса в zora
                // он должен быть больше чем timeout для запроса самого url'а, который будет исполнять zora
                // т.е. мы должны ждать чуть дольше, чем будет ждать zora
                .setRequestTimeout((int) responseTimeout.plusSeconds(1).toMillis())
                .build();
    }
}
