package ru.yandex.direct.zorafetcher;

import java.nio.charset.StandardCharsets;
import java.time.Duration;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.proxy.ProxyServer;

import ru.yandex.direct.tracing.util.TraceUtil;

import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;

/**
 * Создаёт запрос к Zora.
 */
@ParametersAreNonnullByDefault
@SuppressWarnings("SpellCheckingInspection")
public class ZoraOnlineRequestCreator implements ZoraRequestCreator {

    static final String X_YANDEX_SOURCENAME = "X-Yandex-Sourcename";
    static final String X_YANDEX_REQUESTTYPE = "X-Yandex-Requesttype";
    static final String X_YANDEX_REDIRS = "X-Yandex-Redirs";
    static final String REQUEST_TYPE_USERPROXY = "userproxy";
    static final String X_YANDEX_RESPONSE_TIMEOUT = "X-Yandex-Response-Timeout";
    static final String X_YANDEX_NO_CALC = "X-Yandex-NoCalc";
    public static final String X_YANDEX_SERVICE_TICKET = "X-Ya-Service-Ticket";

    private final String zoraSourceName;
    private final ProxyServer zoraProxy;

    public ZoraOnlineRequestCreator(String zoraHost, Integer zoraPort, String zoraSourceName) {
        this.zoraSourceName = zoraSourceName;
        this.zoraProxy = new ProxyServer.Builder(zoraHost, zoraPort).build();
    }

    /**
     * Создать http запрос через прокси Zora.
     * <p>
     * Внимание! Используется режим 'userproxy' предназначенный для запросов, непосредственно порождённых пользователем.
     * Т.е. использовать прокси из скриптов в этом режиме нельзя. https://wiki.yandex-team.ru/Zora/OnlineZora/
     * <p>
     * https://wiki.yandex-team.ru/Zora/UserGuide/#dostuppoprotokoluhttp
     *
     * @param url                    адрес, для которого нужно сделать запрос
     * @param yandexTraceHeaderValue значение для заголовка X-Yandex-Trace {@link TraceUtil#traceToHeader}
     * @param responseTimeout        таймаут на ответ в Zora
     * @return созданный запрос
     */
    @Override
    public Request createGetRequest(String url,
                                    String yandexTraceHeaderValue,
                                    Duration responseTimeout,
                                    String tvmTicket,
                                    boolean followRedirects,
                                    @Nullable String userAgent) {
        RequestBuilder requestBuilder = new RequestBuilder();

        // Zora не умеет https в режиме прокси. Нужно запрашивать http, но устанавливать заголовок `X-Yandex-Use-Https`
        url = url.trim();
        if (url.toLowerCase().startsWith(HTTPS_PROTOCOL + "://")) {
            url = HTTP_PROTOCOL + url.substring(HTTPS_PROTOCOL.length());
            requestBuilder = requestBuilder.setHeader(X_YANDEX_USE_HTTPS, "1");
        }
        requestBuilder.setProxyServer(zoraProxy)
                .setUrl(url)
                .setHeader(X_YANDEX_SOURCENAME, zoraSourceName)
                .setHeader(X_YANDEX_REQUESTTYPE, REQUEST_TYPE_USERPROXY)
                .setHeader(X_YANDEX_TRACE, yandexTraceHeaderValue)
                .setHeader(X_YANDEX_RESPONSE_TIMEOUT, responseTimeout.getSeconds());
        if (followRedirects) {
            requestBuilder
                    .setHeader(X_YANDEX_REDIRS, "1");
        } else {
            requestBuilder
                    .setHeader(X_YANDEX_NO_CALC, "1");
        }
        requestBuilder.setHeader(X_YANDEX_SERVICE_TICKET, tvmTicket);
        return requestBuilder.setCharset(StandardCharsets.UTF_8)
                // это timeout для запроса в zora
                // он должен быть больше чем timeout для запроса самого url'а, который будет исполнять zora
                // т.е. мы должны ждать чуть дольше, чем будет ждать zora
                .setRequestTimeout((int) responseTimeout.plusSeconds(1).toMillis())
                .build();
    }
}
