package ru.yandex.direct.zorafetcher;

import org.asynchttpclient.Response;
import org.asynchttpclient.util.HttpConstants.ResponseStatusCodes;

import ru.yandex.direct.asynchttp.Result;

public abstract class ZoraResponse {

    public static final String X_YANDEX_STATUS = "X-Yandex-Status";
    public static final String X_YANDEX_HTTP_CODE = "X-Yandex-Http-Code";
    public static final String X_YANDEX_ORIG_HTTP_CODE = "X-Yandex-Orig-Http-Code";
    private static final String OK_200_STRING = String.valueOf(ResponseStatusCodes.OK_200);

    protected final Response response;

    public ZoraResponse(Response response) {
        this.response = response;
    }

    public Response getResponse() {
        return response;
    }

    /**
     * Сайт ответил Zora'е {@code "OK 200"}
     */
    public abstract boolean isOk();
    /**
     * Zora ответила, что не ОК с сайтом или с ней.
     *
     * При этом в заголовке {@value X_YANDEX_HTTP_CODE} возвращается код проблемы.
     * Известные коды:
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/zora/zora/proto/statuscodes.proto?rev=3316068#L36">тут</a>
     * и <a href="https://wiki.yandex-team.ru/Robot/KodyOshibok/">тут</a>.
     *
     * @see #isOk()
     */
    public abstract boolean isNotOkByZora();
    /**
     * Возвращает код заголовка {@value #X_YANDEX_HTTP_CODE}
     *
     * @throws NullPointerException если в ответе нет заголовка {@value #X_YANDEX_HTTP_CODE}
     */
    public abstract int getZoraStatusCode();

    public abstract boolean isTimeout(Result<ZoraResponse> result);

    public boolean isRedirect() {
        return response.isRedirected();
    }

    public int getStatusCode() {
        return response.getStatusCode();
    }

    public String getStatusText() {
        return response.getStatusText();
    }

    @Override
    public String toString() {
        return prettyZoraResponse();
    }

    /**
     * Заголовки и statusCode ответа от Zora.
     */
    public String prettyZoraResponse() {
        StringBuilder sb = new StringBuilder();
        sb.append("statusCode=").append(response.getStatusCode()).append('\n')
                .append("headers={\n");
        for (var header : response.getHeaders()) {
            sb.append("  ").append(header.getKey())
                    .append(": ").append(header.getValue())
                    .append('\n');
        }
        sb.append('}');
        return sb.toString();
    }
}
