const fs = require('fs');
const path = require('path');
const tokenator = require('@yandex-int/tokenator');
const VaultClient = require('@yandex-int/vault-client').VaultClient;
const logger = require('./logger');

const SECRETS_PATH = path.resolve(path.join(__dirname, '../../../secrets'));

async function vault(version) {
    let tokenatorTokens = await tokenator(['vault']);
    let vaultClient = new VaultClient(tokenatorTokens.vault);

    let tokens = await vaultClient.getVersion(version);

    return tokens.reduce((memo, token) => {
        memo[token.key] = token.value;

        return memo;
    }, {});
}

module.exports = async function secrets(version) {
    try {
        let secretsVersionPath = path.join(SECRETS_PATH, version);
        let tokens;

        if (!fs.existsSync(secretsVersionPath)) {
            logger.info(`Started downloading secrets '${version}'. Please wait...`);

            tokens = await vault(version);

            if (tokens) {
                logger.info(`Secrets '${version}' downloaded successfully!`);
            }

            if (!fs.existsSync(SECRETS_PATH)) {
                fs.mkdirSync(SECRETS_PATH, 0o755);
            }

            fs.writeFileSync(secretsVersionPath, JSON.stringify(tokens, null, 4));
        } else {
            tokens = JSON.parse(fs.readFileSync(secretsVersionPath, 'utf8'));
        }

        return tokens;
    } catch (error) {
        if (error.message.includes('self signed certificate in certificate chain')) {
            logger.error('Internal certificate YandexInternalRootCA.pem do not installed properly.');
            logger.error('Please run `./tools/prepare-internal-root-ca.sh` to fix it');
            logger.error(error);
        } else {
            logger.error('Access denied or unexpected error. Please call @next0 or @ssamsonov for help.');
            logger.error(error);
        }

        throw error;
    }
};
