import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { CloseCircleOutlined, QuestionCircleOutlined } from '@ant-design/icons';
import { Input, Tag, Tooltip } from 'antd';

import { ColumnSchema } from 'schema/logs/LogSourceSchema';

import { cn } from 'helpers/className/className';
import { useHelper } from 'helpers/common/useHelper/useHelper';

import { getLogConfig } from 'selectors/logs/getLogConfig/getLogConfig';
import { getUIFilterFormConditions } from 'selectors/ui/getUIFilterFormConditions/getUIFilterFormConditions';
import { getUIFilterFormLogType } from 'selectors/ui/getUIFilterFormLogType/getUIFilterFormLogType';

import { changeUIFilterFormConditions } from 'services/filterForm/changeUIFilterFormConditions/changeUIFilterFormConditions';
import { replaceUIFilterFormCondition } from 'services/filterForm/replaceUIFilterFormCondition/replaceUIFilterFormCondition';
import { updateUIFilterForm } from 'services/filterForm/updateUIFilterForm/updateUIFilterForm';

import { FilterName } from 'components/Filters/FilterName/FilterName';

import 'antd/dist/antd.css';
import './FilterConditions.css';

const { CheckableTag } = Tag;

const cls = cn('filter-conditions');

interface FilterConditionsProps {
    className?: string;
    onPressConditionEnter: () => void
}

interface ColumnItemProps {
    column: ColumnSchema;
    checked: boolean;
    onChange: (value: string) => void;
}

const ColumnItem = function ColumnItem({ column, checked, onChange }: ColumnItemProps) {
    const onCheck = useCallback(() => onChange(column.name), [column.name, onChange]);

    return (
        <CheckableTag className={cls('checkbox', { checked: checked })} onChange={onCheck} checked={checked}>
            {column.name}
        </CheckableTag>
    );
};

const FilterConditionItem = function ({
    name,
    value,
    onDelete,
    onChange,
    onPressConditionEnter,
}: {
    name: string;
    value: string;
    onDelete: (column: string) => void;
    onChange: (column: string, value: string) => void;
    onPressConditionEnter: () => void
}) {
    const onDeleteClick = useCallback(() => onDelete(name), [name, onDelete]);
    const onChangeHandle = useCallback((event) => onChange(name, event.target.value), [name, onChange]);

    return (
        <div key={name} className={cls('line')}>
            <FilterName>{name}</FilterName>

            <Input id={'logField-' + name}
                   className={cls('input')}
                   value={value}
                   autoFocus={true}
                   onChange={onChangeHandle}
                   onPressEnter={onPressConditionEnter}/>

            <CloseCircleOutlined className={cls('close-icon')} onClick={onDeleteClick} />
        </div>
    );
};

export const FilterConditions = React.memo(function (props: FilterConditionsProps) {
    const dispatch = useDispatch();
    const log = useSelector(getUIFilterFormLogType);
    const conditions = useSelector(getUIFilterFormConditions);
    const currentLogConfig = useHelper(getLogConfig, log);
    const onChange = useCallback(
        (column: string) => {
            dispatch(changeUIFilterFormConditions({ column }));
        },
        [dispatch],
    );
    const onDeleteCondition = useCallback(
        (column: string) => {
            let newConditions = { ...conditions };
            delete newConditions[column];
            dispatch(
                updateUIFilterForm({
                    conditions: newConditions,
                }),
            );
        },
        [conditions, dispatch],
    );
    const onChangeConditionValue = useCallback(
        (column: string, value: string) => {
            dispatch(replaceUIFilterFormCondition({ column, value }));
        },
        [dispatch],
    );

    const tooltipTitle = useMemo(
        () => (
            <>
                Simple expressions supported:
                <br />
                {'<'}5
                <br />
                {'>'}=2.56
                <br />
                show%
                <br />
                !%xxx%
                <br />
                <a
                    target={'_blank'}
                    href={'https://docs.yandex-team.ru/direct-dev/reference/logviewer/filter-query-language'}
                >
                    Details
                </a>
            </>
        ),
        [],
    );

    if (!log || !currentLogConfig) {
        return null;
    }

    return (
        <div className={cls(null, [props.className])}>
            <div className={cls('line')}>
                <FilterName>
                    <Tooltip title={tooltipTitle}>
                        <QuestionCircleOutlined className={cls('help-icon')} /> Filter conditions:
                    </Tooltip>
                </FilterName>

                <div className={cls('checkboxes')}>
                    {currentLogConfig.columns
                        .filter((column) => !column.virtual)
                        .map((column) => {
                            return (
                                <ColumnItem
                                    key={column.name}
                                    onChange={onChange}
                                    checked={Boolean(conditions && conditions[column.name] !== undefined)}
                                    column={column}
                                />
                            );
                        })}
                </div>
            </div>

            {Object.keys(conditions).length > 0 &&
                Object.keys(conditions).map((formName) => {
                    return (
                        <FilterConditionItem
                            onChange={onChangeConditionValue}
                            onDelete={onDeleteCondition}
                            key={formName}
                            name={formName}
                            value={conditions[formName]}
                            onPressConditionEnter={props.onPressConditionEnter}
                        />
                    );
                })}
        </div>
    );
});
