import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { CopyOutlined } from '@ant-design/icons';
import { StopOutlined } from '@ant-design/icons';
import { Alert, Button, Card, Checkbox, DatePicker, Select } from 'antd';
import moment, { Moment } from 'moment';
import { RangeValue } from 'rc-picker/lib/interface';

import { cn } from 'helpers/className/className';
import { useHelper } from 'helpers/common/useHelper/useHelper';

import { getLogsDataQueryId } from 'selectors/logs/getLogsDataQueryId/getLogsDataQueryId';
import { isLogsConfigLoading } from 'selectors/logs/isLogsConfigLoading/isLogsConfigLoading';
import { isLogsDataLoading } from 'selectors/logs/isLogsDataLoading/isLogsDataLoading';
import { isShowAllTraceIdVisible } from 'selectors/logs/isShowAllTraceIdVisible/isShowAllTraceIdVisible';
import { getUIFilterFormError } from 'selectors/ui/getUIFilterFormError/getUIFilterFormError';
import { getUIFilterFormFrom } from 'selectors/ui/getUIFilterFormFrom/getUIFilterFormFrom';
import { getUIFilterFormLogType } from 'selectors/ui/getUIFilterFormLogType/getUIFilterFormLogType';
import { getUIFilterFormPageSize } from 'selectors/ui/getUIFilterFormPageSize/getUIFilterFormPageSize';
import { getUIFilterFormReverseOrder } from 'selectors/ui/getUIFilterFormReverseOrder/getUIFilterFormReverseOrder';
import { getUIFilterFormShowTraceIdRelated } from 'selectors/ui/getUIFilterFormShowTraceIdRelated/getUIFilterFormShowTraceIdRelated';
import { getUIFilterFormTo } from 'selectors/ui/getUIFilterFormTo/getUIFilterFormTo';
import { getUIPreparingToShareQueryId } from 'selectors/ui/getUIPreparingToShareQueryId/getUIPreparingToShareQueryId';

import { updateUIFilterForm } from 'services/filterForm/updateUIFilterForm/updateUIFilterForm';
import { shareShortLinkByQueryId } from 'services/shareShortLinkByQueryId/shareShortLinkByQueryId';
import { showLogs } from 'services/showLogs/showLogs';

import { DATE_SHOW_LOGS_API_FORMAT } from 'enums/date';

import { FilterConditions } from 'components/Filters/FilterConditions/FilterConditions';
import { FilterName } from 'components/Filters/FilterName/FilterName';
import { LogSource } from 'components/Filters/LogSource/LogSource';
import { ShowStats } from 'components/Filters/ShowStats/ShowStats';

import 'antd/dist/antd.css';
import './Filters.css';

const { RangePicker } = DatePicker;
const { Option } = Select;

const cls = cn('filters');

interface FiltersProps {
    className?: string;
    abortFetchLogsRequest: () => void;
    getFetchLogsRequestSignal: () => AbortSignal;
}

const ranges: Dict<Exclude<RangeValue<Moment>, null>> = {
    'Last Hour': [moment().subtract(1, 'hours'), moment().endOf('day')],
    Today: [moment().startOf('day'), moment().endOf('day')],
    Yesterday: [moment().subtract(1, 'days').startOf('day'), moment().subtract(1, 'days').endOf('day')],
    'Last 3 Days': [moment().subtract(2, 'days').startOf('day'), moment().endOf('day')],
    'Last Week': [moment().subtract(6, 'days').startOf('day'), moment().endOf('day')],
    'Last Month': [moment().subtract(1, 'months').startOf('day'), moment().endOf('day')],
    'Last 6 Months': [moment().subtract(6, 'months').startOf('day'), moment().endOf('day')],
    'Last Year': [moment().subtract(1, 'years').startOf('day'), moment().endOf('day')],
};

const pageSizeOptions = [
    {
        value: 100,
        name: '100 / page',
    },
    {
        value: 200,
        name: '200 / page',
    },
    {
        value: 500,
        name: '500 / page',
    },
    {
        value: 1000,
        name: '1 000 / page',
    },
];

function getDatePickerValue(from?: string, to?: string) {
    if (!from || !to) {
        return [];
    }

    return [moment(from, DATE_SHOW_LOGS_API_FORMAT), moment(to, DATE_SHOW_LOGS_API_FORMAT)];
}

export const Filters = React.memo(function (props: FiltersProps) {
    const dispatch = useDispatch();
    const { getFetchLogsRequestSignal, abortFetchLogsRequest } = props;
    const log = useSelector(getUIFilterFormLogType);
    const pageSize = useSelector(getUIFilterFormPageSize);
    const preparingToShareQueryId = useSelector(getUIPreparingToShareQueryId);
    const reverseOrder = useSelector(getUIFilterFormReverseOrder);
    const showTraceIdRelated = useSelector(getUIFilterFormShowTraceIdRelated);
    const showAllTraceIdVisible = useHelper(isShowAllTraceIdVisible, log);
    const from = useSelector(getUIFilterFormFrom);
    const queryId = useSelector(getLogsDataQueryId);
    const to = useSelector(getUIFilterFormTo);
    const datePickerValue = useMemo(() => getDatePickerValue(from, to), [from, to]);
    const error = useSelector(getUIFilterFormError);
    const logsConfigLoading = useSelector(isLogsConfigLoading);
    const logsDataLoading = useSelector(isLogsDataLoading);
    const isLoading = logsConfigLoading || logsDataLoading;
    const linkIcon = useMemo(() => <CopyOutlined />, []);
    const onShortLinkClick = useCallback(() => {
        if (queryId) {
            dispatch(shareShortLinkByQueryId({ queryId }));
        }
    }, [dispatch, queryId]);

    const onShowClick = useCallback(() => {
        dispatch(showLogs({ signal: getFetchLogsRequestSignal() }));
    }, [dispatch, getFetchLogsRequestSignal]);

    const onStopClick = useCallback(() => {
        abortFetchLogsRequest();
    }, [abortFetchLogsRequest]);

    const onChangePageSize = useCallback(
        (value: number) => {
            dispatch(
                updateUIFilterForm({
                    limit: value,
                }),
            );
        },
        [dispatch],
    );
    const onChangeReverseOrder = useCallback(
        (e) => {
            dispatch(
                updateUIFilterForm({
                    reverseOrder: e.target.checked,
                }),
            );
        },
        [dispatch],
    );
    const onChangeShowTraceIdRelated = useCallback(
        (e) => {
            dispatch(
                updateUIFilterForm({
                    showTraceIdRelated: e.target.checked,
                }),
            );
        },
        [dispatch],
    );
    const onChangeDateRange = useCallback(
        ([from, to]) => {
            let fromStr = from.format(DATE_SHOW_LOGS_API_FORMAT);
            let toStr = to.format(DATE_SHOW_LOGS_API_FORMAT);

            dispatch(
                updateUIFilterForm({
                    from: fromStr,
                    to: toStr,
                }),
            );
        },
        [dispatch],
    );

    const copyShortLinkBtn = useMemo(() => {
        if (!queryId) {
            return null;
        }

        const isLoading = preparingToShareQueryId === queryId;

        return (
            <Button
                type="link"
                loading={isLoading}
                className={cls('button')}
                icon={linkIcon}
                onClick={onShortLinkClick}
            >
                Short link
            </Button>
        );
    }, [preparingToShareQueryId, queryId, linkIcon, onShortLinkClick]);

    return (
        <Card className={cls(null, [props.className])}>
            <LogSource className={cls('line')} />

            <div className={cls('line')}>
                <FilterName>Range:</FilterName>

                <RangePicker
                    allowClear={false}
                    id={'range'}
                    size={'large'}
                    showTime
                    ranges={ranges}
                    onChange={onChangeDateRange}
                    value={datePickerValue as [Moment, Moment]}
                />
            </div>

            <FilterConditions onPressConditionEnter={onShowClick} className={cls('line')} />

            <ShowStats className={cls('buttons')} />

            <div className={cls('buttons')}>
                <Button
                    loading={isLoading}
                    className={cls('submit')}
                    type="primary"
                    size={'large'}
                    onClick={onShowClick}
                    disabled={!log}
                >
                    Show
                </Button>

                {copyShortLinkBtn}

                <Select style={{ width: 125 }} value={pageSize} onChange={onChangePageSize}>
                    {pageSizeOptions.map((option) => (
                        <Option value={option.value} key={option.value}>
                            {option.name}
                        </Option>
                    ))}
                </Select>

                <Checkbox checked={reverseOrder} onChange={onChangeReverseOrder} className={cls('button')}>
                    reverse order
                </Checkbox>
                {showAllTraceIdVisible && (
                    <Checkbox
                        checked={showTraceIdRelated}
                        onChange={onChangeShowTraceIdRelated}
                        className={cls('button')}
                    >
                        search all trace id
                    </Checkbox>
                )}
                {isLoading &&
                    <Button
                        className={cls('stop')}
                        icon={<StopOutlined />}
                        type="primary"
                        size={'large'}
                        onClick={onStopClick}
                    >
                        Stop
                    </Button>
                }
            </div>

            {error && <Alert className={cls('error')} message={error} type="error" />}
        </Card>
    );
});
