import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch } from 'react-redux';
import { DownOutlined } from '@ant-design/icons';
import { Button, Checkbox } from 'antd';

import { cn } from 'helpers/className/className';

import { updateUIFilterForm } from 'services/filterForm/updateUIFilterForm/updateUIFilterForm';

import { ColumnsSelect } from 'components/ColumnsSelect/ColumnsSelect';

import './FiltersColumns.css';

interface ColumnsSelectProps {
    className?: string;
    options: string[];
    defaultOptions: string[];
    value?: string[];
    disabled?: boolean;
}

const cls = cn('filters-columns');

function getInvertedColumns(options: string[], value: string[]) {
    const valueSet = new Set(value);

    return options.filter((option) => !valueSet.has(option));
}

function getButtonText(options: string[], value?: string[]): string {
    if (!value?.length) {
        return 'Columns';
    }

    if (value.length === options.length) {
        return 'All columns';
    }

    if (value.length === 1) {
        return value[0];
    }

    return value.length + ' columns';
}

export const FiltersColumns = React.memo(function FiltersColumns({
    options,
    className,
    disabled,
    value,
    defaultOptions,
}: ColumnsSelectProps) {
    const dispatch = useDispatch();
    const indeterminateChecked = value && value.length > 0 && value.length < options.length;
    const allChecked = value && value.length === options.length;
    const onChangeColumn = useCallback(
        (fields: string[]) => {
            dispatch(
                updateUIFilterForm({
                    fields,
                }),
            );
        },
        [dispatch],
    );
    const onCheckAll = useCallback(
        (event) => {
            dispatch(
                updateUIFilterForm({
                    fields: event.target.checked ? options : [],
                }),
            );
        },
        [dispatch, options],
    );
    const onInvertClick = useCallback(() => {
        dispatch(
            updateUIFilterForm({
                fields: getInvertedColumns(options, value || []),
            }),
        );
    }, [dispatch, options, value]);
    const onDefaultClick = useCallback(() => {
        dispatch(
            updateUIFilterForm({
                fields: defaultOptions,
            }),
        );
    }, [defaultOptions, dispatch]);
    const title = useMemo(() => {
        return (
            <div className={cls('title')}>
                <Checkbox indeterminate={indeterminateChecked} onChange={onCheckAll} checked={allChecked}>
                    All
                </Checkbox>
                <div className={cls('title-right')}>
                    <a onClick={onDefaultClick}>Default</a>
                    <a onClick={onInvertClick}>Invert</a>
                </div>
            </div>
        );
    }, [allChecked, indeterminateChecked, onCheckAll, onDefaultClick, onInvertClick]);

    return (
        <ColumnsSelect
            title={title}
            onChangeColumn={onChangeColumn}
            className={cls(null, [className])}
            options={options}
            value={value}
            disabled={disabled}
        >
            <Button disabled={disabled} size={'large'}>
                {getButtonText(options, value)} <DownOutlined />
            </Button>
        </ColumnsSelect>
    );
});
