import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { Select } from 'antd';

import { LogsConfigSchema } from 'schema/logs/LogsConfigSchema';
import { LogSourceSchema } from 'schema/logs/LogSourceSchema';

import { cn } from 'helpers/className/className';
import { useHelper } from 'helpers/common/useHelper/useHelper';
import { getDefaultFieldsForLogType } from 'helpers/form/getDefaultFieldsForLogType/getDefaultFieldsForLogType';

import { getLogConfig } from 'selectors/logs/getLogConfig/getLogConfig';
import { getLogsConfig } from 'selectors/logs/getLogsConfig/getLogsConfig';
import { getUIFilterFormFields } from 'selectors/ui/getUIFilterFormFields/getUIFilterFormFields';
import { getUIFilterFormLogType } from 'selectors/ui/getUIFilterFormLogType/getUIFilterFormLogType';

import { updateUIFilterForm } from 'services/filterForm/updateUIFilterForm/updateUIFilterForm';

import { FilterName } from 'components/Filters/FilterName/FilterName';
import { FiltersColumns } from 'components/Filters/FiltersColumns/FiltersColumns';

import 'antd/dist/antd.css';
import './LogSource.css';

const cls = cn('log-source');

interface LogSourceProps {
    className?: string;
}

function getLogsOptions(
    logsConfig: Optional<LogsConfigSchema>,
): Array<{
    label: string;
    value: string;
}> {
    if (!logsConfig) {
        return [];
    }

    return Object.keys(logsConfig).map((logName) => {
        const log = logsConfig[logName];

        return {
            value: logName,
            label: [logName, log.desc].filter(Boolean).join(' - '),
        };
    });
}

function getColumnsOptions(config: Optional<LogSourceSchema>): string[] {
    if (!config) {
        return [];
    }

    return config.columns.filter(column => !column.strictlyHidden).map((column) => column.name);
}

export const LogSource = React.memo(function LogSource(props: LogSourceProps) {
    const dispatch = useDispatch();
    const log = useSelector(getUIFilterFormLogType);
    const currentLogConfig = useHelper(getLogConfig, log);
    const fields = useSelector(getUIFilterFormFields);
    const logsConfig = useSelector(getLogsConfig);
    const logsOptions = useMemo(() => getLogsOptions(logsConfig), [logsConfig]);
    const columnsOptions = useMemo(() => getColumnsOptions(currentLogConfig), [currentLogConfig]);
    const currentDefaultOptions = useMemo(() => getDefaultFieldsForLogType(log, logsConfig), [log, logsConfig]);

    const onChangeLog = useCallback(
        (value: string) => {
            dispatch(
                updateUIFilterForm({
                    logType: value,
                    fields: getDefaultFieldsForLogType(value, logsConfig),
                    conditions: {},
                }),
            );
        },
        [dispatch, logsConfig],
    );

    return (
        <div className={cls(null, [props.className])}>
            <FilterName>Log source:</FilterName>

            <Select
                id={'log_source'}
                showSearch
                style={{ width: 424 }}
                placeholder="Выберите место логов"
                optionFilterProp="label"
                size={'large'}
                value={log}
                onChange={onChangeLog}
                dropdownMatchSelectWidth={false}
                options={logsOptions}
                allowClear
                filterOption
            />
            <FiltersColumns
                defaultOptions={currentDefaultOptions}
                className={cls('columns')}
                value={fields}
                options={columnsOptions}
                disabled={!log}
            />
        </div>
    );
});
