import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { CopyOutlined } from '@ant-design/icons';
import { Button, Spin, Typography } from 'antd';
import moment from 'moment';

import { HistoryItemSchema } from 'schema/history/HistorySchema';

import { cn } from 'helpers/className/className';

import { getUIPreparingToShareQueryId } from 'selectors/ui/getUIPreparingToShareQueryId/getUIPreparingToShareQueryId';

import { fetchLogsByQueryId } from 'services/fetchLogsByQueryId/fetchLogsByQueryId';
import { shareShortLinkByQueryId } from 'services/shareShortLinkByQueryId/shareShortLinkByQueryId';

import { DATE_SHOW_HISTORY_FORMAT } from 'enums/date';

import { HistoryStatusIcon } from 'components/HistoryPanel/HistoryStatusIcon/HistoryStatusIcon';

import './HistoryPanelItem.css';
const cls = cn('history-panel-item');
const { Link, Text } = Typography;

interface HistoryPanelItemsProps extends HistoryItemSchema {
    className?: string;
}

export const HistoryPanelItem = React.memo((props: HistoryPanelItemsProps) => {
    const { status, name, startTime, queryId } = props;
    const preparingToShareQueryId = useSelector(getUIPreparingToShareQueryId);
    const dispatch = useDispatch();
    const linkIcon = useMemo(() => {
        if (preparingToShareQueryId && preparingToShareQueryId === queryId) {
            return <Spin size={'small'} />;
        }

        return <CopyOutlined />;
    }, [preparingToShareQueryId, queryId]);

    const onItemClick = useCallback(() => {
        if (!preparingToShareQueryId) {
            dispatch(fetchLogsByQueryId({ queryId }));
        }
    }, [queryId, dispatch, preparingToShareQueryId]);

    const onShortLinkClick = useCallback(() => {
        dispatch(shareShortLinkByQueryId({ queryId }));
    }, [queryId, dispatch]);

    return (
        <div className={cls(undefined, [props.className])}>
            <div className={cls('main')}>
                <div className={cls('status-icon')}>
                    <HistoryStatusIcon status={status} />
                </div>
                <Link className={cls('link')} ellipsis onClick={onItemClick}>
                    {name}
                </Link>
                <div className={cls('buttons-block')}>
                    <Button
                        size={'small'}
                        type="link"
                        className={cls('copy-button')}
                        icon={linkIcon}
                        onClick={onShortLinkClick}
                    />
                </div>
            </div>
            <div className={cls('comment')}>
                <Text type="secondary">{moment(startTime).format(DATE_SHOW_HISTORY_FORMAT)}</Text>
            </div>
        </div>
    );
});
