import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { SettingOutlined } from '@ant-design/icons';
import { Checkbox } from 'antd';

import { cn } from 'helpers/className/className';

import {getSavedFormFields} from 'selectors/logs/getSavedFormFields/getSavedFormFields';
import { getUIHiddenColumns } from 'selectors/ui/getUIHiddenColumns/getUIHiddenColumns';

import { updateUI } from 'services/ui/updateUI/updateUI';
import { updateUIHiddenColumns } from 'services/ui/updateUIHiddenColumns/updateUIHiddenColumns';

import { ColumnsSelect } from 'components/ColumnsSelect/ColumnsSelect';

import './LogsColumns.css';

interface ColumnsSelectProps {
    className?: string;
}

const cls = cn('logs-columns');

function getValue(hiddenColumns: Dict<boolean>, fields: string[]): string[] {
    return fields.filter(field => !hiddenColumns[field]);
}

const DEFAULT_FIELDS: string[] = [];

export const LogsColumns = React.memo(function LogsColumns({ className }: ColumnsSelectProps) {
    const dispatch = useDispatch();
    const fields = useSelector(getSavedFormFields) || DEFAULT_FIELDS;
    const hiddenColumns = useSelector(getUIHiddenColumns);
    const value = useMemo(() => getValue(hiddenColumns, fields), [fields, hiddenColumns]);
    const indeterminateChecked = value && value.length > 0 && value.length < fields.length;
    const allChecked = value && value.length === fields.length;
    const onChangeColumn = useCallback(
        (newFields: string[]) => {
            dispatch(updateUI({
                hiddenColumns: fields.filter(field => !newFields.includes(field)).reduce((result, field) => {
                    result[field] = true;

                    return result;
                }, {} as Dict<boolean>)
            }));
        },
        [dispatch, fields],
    );
    const onCheckAll = useCallback(
        (event) => {
            if (event.target.checked) {
                dispatch(updateUI({
                    hiddenColumns: {}
                }));

                return;
            }

            dispatch(
                updateUIHiddenColumns(
                    fields.reduce((result, current) => {
                        result[current] = true;

                        return result;
                    }, {} as Dict<boolean>)
                ),
            );
        },
        [dispatch, fields],
    );
    const onResetClick = useCallback(() => {
        dispatch(updateUI({
            hiddenColumns: {}
        }));
    }, [dispatch]);
    const title = useMemo(() => {
        return (
            <div className={cls('title')}>
                <Checkbox indeterminate={indeterminateChecked} onChange={onCheckAll} checked={allChecked}>
                    Column Display
                </Checkbox>
                <div className={cls('title-right')}>
                    <a onClick={onResetClick}>Reset</a>
                </div>
            </div>
        );
    }, [allChecked, indeterminateChecked, onCheckAll, onResetClick]);

    return (
        <ColumnsSelect
            title={title}
            onChangeColumn={onChangeColumn}
            className={cls(null, [className])}
            options={fields}
            value={value}
            placement={'bottomRight'}
        >
            <SettingOutlined className={cls('icon')} />
        </ColumnsSelect>
    );
});
