import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import {
    ArrowUpOutlined,
    CopyOutlined,
    EllipsisOutlined,
    EyeInvisibleOutlined,
    SelectOutlined,
} from '@ant-design/icons';
import { Alert, Button, Card, Table } from 'antd';

import { LogDataRowItem } from 'schema/logs/LogDataSchema';
import { LogSourceSchema } from 'schema/logs/LogSourceSchema';

import { cn } from 'helpers/className/className';
import { copyToClipboard } from 'helpers/common/copyToClipboard/copyToClipboard';
import { useHelper } from 'helpers/common/useHelper/useHelper';

import { getLogConfig } from 'selectors/logs/getLogConfig/getLogConfig';
import { getLogs } from 'selectors/logs/getLogs/getLogs';
import { getLogsData } from 'selectors/logs/getLogsData/getLogsData';
import { getLogsDataError } from 'selectors/logs/getLogsDataError/getLogsDataError';
import { getLogsDataTotalCount } from 'selectors/logs/getLogsDataTotalCount/getLogsDataTotalCount';
import { getSavedFormFields } from 'selectors/logs/getSavedFormFields/getSavedFormFields';
import { getSavedFormLimit } from 'selectors/logs/getSavedFormLimit/getSavedFormLimit';
import { getSavedFormLogType } from 'selectors/logs/getSavedFormLogType/getSavedFormLogType';
import { getSavedFormOffset } from 'selectors/logs/getSavedFormOffset/getSavedFormOffset';
import { getSavedFormShowStats } from 'selectors/logs/getSavedFormShowStats/getSavedFormShowStats';
import { isLogsDataLoading } from 'selectors/logs/isLogsDataLoading/isLogsDataLoading';
import { getUIFilterFormUncollapsedValues } from 'selectors/ui/getUIFilterFormUncollapsedValues/getUIFilterFormUncollapsedValues';
import { getUIHiddenColumns } from 'selectors/ui/getUIHiddenColumns/getUIHiddenColumns';
import { getUIViewInfo } from 'selectors/ui/getUIViewInfo/getUIViewInfo';

import { changeCollapse } from 'services/collapseInfo/changeCollapse/changeCollapse';
import { addUIFilterFormCondition } from 'services/filterForm/addUIFilterFormCondition/addUIFilterFormCondition';
import { showLogsPage } from 'services/showLogsPage/showLogsPage';
import { updateUIHiddenColumns } from 'services/ui/updateUIHiddenColumns/updateUIHiddenColumns';

import { LogsToolbar } from 'components/LogsTable/LogsToolbar/LogsToolbar';
import { QuickSearchItems } from 'components/LogsTable/QuickSearchItems/QuickSearchItems';

import './LogsTable.css';

const cls = cn('logs-table');
const scrollConfiguration = {
    x: '100%',
};

interface LogsTableProps {
    className?: string;
    getFetchLogsRequestSignal: () => AbortSignal;
}

interface ColumnType {
    key: string;
    title: React.ReactNode;
    dataIndex: string;
    className?: string;
    render?: (text: string, record: { key: number | string }) => React.ReactNode;
}

interface CellButtonsProps {
    value: string;
    column: string;
    index: number | string;
    hasUncollapseButton?: boolean;
    isTraceId?: boolean;
    isReqId?: boolean;
}

function getHttpStatusMods(status: number): string {
    if (status >= 200 && status < 300) {
        return 'ok';
    }

    if (status >= 400 && status < 500) {
        return 'warn';
    }

    if (status >= 500 && status < 600) {
        return 'error';
    }

    return 'info';
}

const CellButtons = React.memo(
    ({ column, value, index, hasUncollapseButton, isTraceId, isReqId }: CellButtonsProps) => {
        const dispatch = useDispatch();
        let elemMods = undefined;
        let viewInfo = useSelector(getUIViewInfo);
        let isUncollapseByDefault = useSelector(getUIFilterFormUncollapsedValues);

        let cellId = index + '_' + column;
        const onCollapseClick = useCallback(() => {
            dispatch(changeCollapse({ cellId }));
        }, [cellId, dispatch]);

        let runLink = value && value.webApiGrid;
        hasUncollapseButton = value && (value.collapse !== undefined && value.full !== undefined);
        let displayValue: any;
        if (hasUncollapseButton) {
            let cellViewInfo = viewInfo[cellId];

            if (cellViewInfo === undefined) {
                if (isUncollapseByDefault) {
                    cellViewInfo = 'full';
                } else {
                    cellViewInfo = 'collapse';
                }
            }
            displayValue = value[cellViewInfo];
        } else if (value && value.full !== undefined) {
            displayValue = value.full;
        } else {
            displayValue = value;
        }

        let fullValue: string;
        if (value && value.full !== undefined) {
            fullValue = value.full;
        } else {
            fullValue = value;
        }

        const onCopyClick = useCallback(() => {
            copyToClipboard(fullValue);
        }, [fullValue]);
        const onFilterClick = useCallback(() => {
            dispatch(addUIFilterFormCondition({ column, value: displayValue }));
        }, [column, dispatch, displayValue]);

        if (column === 'log_level') {
            elemMods = {
                [column]: displayValue.toLocaleLowerCase(),
            };
        }

        if (column === 'http_status' && typeof displayValue === 'number') {
            elemMods = {
                [column]: getHttpStatusMods(displayValue),
            };
        }

        return (
            <div className={cls('cell', elemMods)}>
                {(isTraceId || isReqId) && (
                    <QuickSearchItems
                        className={cls('search-icon')}
                        column={isTraceId ? 'trace_id' : 'reqid'}
                        value={displayValue}
                    />
                )}

                {displayValue}
                <div className={cls('cell-buttons')}>
                    {runLink && (
                        <Button
                            type="link"
                            className={cls('button')}
                            icon={<SelectOutlined />}
                            href={runLink}
                            target={'_blank'}
                        >
                            Run
                        </Button>
                    )}
                    <Button
                        title={'Add to filter'}
                        className={cls('cell-button')}
                        icon={<ArrowUpOutlined />}
                        size={'small'}
                        onClick={onFilterClick}
                    />
                    <Button
                        title={'Copy cell value'}
                        onClick={onCopyClick}
                        className={cls('cell-button')}
                        icon={<CopyOutlined />}
                        size={'small'}
                    />
                    {hasUncollapseButton && (
                        <Button
                            title={'Uncollapsed value'}
                            onClick={onCollapseClick}
                            className={cls('cell-button')}
                            icon={<EllipsisOutlined />}
                            size={'small'}
                        />
                    )}
                </div>
            </div>
        );
    },
);

const CellColumn = React.memo(({ field, onHideColumn }: { field: string; onHideColumn: (field: string) => void }) => {
    const onHide = useCallback(() => {
        onHideColumn(field);
    }, [field, onHideColumn]);

    return (
        <div className={cls('title')}>
            {field}
            <EyeInvisibleOutlined onClick={onHide} className={cls('title-icon')} />
        </div>
    );
});

function getColumns(
    fields: string[],
    onHideColumn: (field: string) => void,
    hiddenColumns: Dict<boolean>,
    showStats?: boolean,
    logConfig?: LogSourceSchema,
): ColumnType[] {
    const columns: ColumnType[] = fields
        .filter((field) => !hiddenColumns[field])
        .map((field) => {
            return {
                key: field,
                title: <CellColumn field={field} onHideColumn={onHideColumn} />,
                dataIndex: field,
                className: cls(field),
                render: (value, row) => {
                    return (
                        <CellButtons
                            column={field}
                            value={value}
                            index={row.key}
                            isTraceId={field === logConfig?.traceIdColumn}
                            isReqId={field === logConfig?.reqidColumn}
                        />
                    );
                },
            };
        });

    columns.unshift({
        title: '№',
        dataIndex: 'key',
        key: 'key',
        className: cls('key'),
    });

    if (showStats) {
        columns.push({
            title: 'count',
            dataIndex: 'count',
            key: 'count',
            className: cls('count'),
        });
    }

    return columns;
}

type DataSource = Record<'key', string | number> & Dict<string | number>;

function getDataSource(
    data: LogDataRowItem[],
    hiddenColumns: Dict<boolean>,
    fields: string[] = [],
    offset: number,
    showStats?: boolean,
): DataSource[] {
    const columnsFields = ['key'].concat(fields);

    if (showStats) {
        columnsFields.push('count');
    }

    return data.map((item, dataIndex) => {
        return columnsFields.reduce((result, column, index) => {
            if (column === 'key') {
                result.key = offset + dataIndex + 1;

                return result;
            }

            if (hiddenColumns[column]) {
                return result;
            }

            result[column] = item[index - 1];
            return result;
        }, {} as DataSource);
    });
}

function getPaginationSettings(pageSize: number, onChangePage: (page: number) => void, offset: number, total?: number) {
    return {
        pageSize,
        showSizeChanger: false,
        total,
        current: offset / pageSize + 1,
        onChange: onChangePage,
    };
}

export const LogsTable = function LogsTable(props: LogsTableProps) {
    const dispatch = useDispatch();
    const fields = useSelector(getSavedFormFields);
    const showStats = useSelector(getSavedFormShowStats);
    const logs = useSelector(getLogs);
    const logType = useSelector(getSavedFormLogType);
    const logConfig = useHelper(getLogConfig, logType);
    const logsData = useSelector(getLogsData);
    const logsDataError = useSelector(getLogsDataError);
    const totalCount = useSelector(getLogsDataTotalCount);
    const logsDataLoading = useSelector(isLogsDataLoading);
    const pageSize = useSelector(getSavedFormLimit);
    const offset = useSelector(getSavedFormOffset);
    const hiddenColumns = useSelector(getUIHiddenColumns);

    const { getFetchLogsRequestSignal } = props;
    const onChangePage = useCallback(
        (page) => {
            dispatch(showLogsPage({ page, signal: getFetchLogsRequestSignal() }));
        },
        [dispatch, getFetchLogsRequestSignal],
    );
    const onHideColumn = useCallback(
        (field) => {
            dispatch(updateUIHiddenColumns({ [field]: true }));
        },
        [dispatch],
    );

    const paginationSettings = useMemo(() => getPaginationSettings(pageSize, onChangePage, offset, totalCount), [
        offset,
        onChangePage,
        pageSize,
        totalCount,
    ]);
    const columns = useMemo(() => getColumns(fields || [], onHideColumn, hiddenColumns, showStats, logConfig), [
        fields,
        onHideColumn,
        hiddenColumns,
        showStats,
        logConfig,
    ]);
    const dataSource = useMemo(() => getDataSource(logsData || [], hiddenColumns, fields, offset, showStats), [
        fields,
        hiddenColumns,
        logsData,
        offset,
        showStats
    ]);

    if (!logs) {
        return null;
    }

    return (
        <Card className={cls(null, [props.className])}>
            {!logsDataLoading && <LogsToolbar className={cls('toolbar')} />}
            {logsDataError && <Alert className={cls('error')} message={logsDataError} type="error" />}
            <Table
                rowClassName={cls('row')}
                scroll={scrollConfiguration}
                loading={logsDataLoading}
                className={cls('table')}
                dataSource={dataSource}
                columns={columns}
                pagination={paginationSettings}
                bordered
                size={'small'}
            />
        </Card>
    );
};
