import * as React from 'react';
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { DownloadOutlined, SelectOutlined } from '@ant-design/icons';
import { Button, Checkbox } from 'antd';

import { cn } from 'helpers/className/className';
import { getDownloadLogsUrl } from 'helpers/url/getDownloadLogsUrl/getDownloadLogsUrl';
import { getYqlUrl } from 'helpers/url/getYqlUrl/getYqlUrl';

import { getLogsSql } from 'selectors/logs/getLogsSql/getLogsSql';
import { getSavedForm } from 'selectors/logs/getSavedForm/getSavedForm';
import { getUIFilterFormUncollapsedValues } from 'selectors/ui/getUIFilterFormUncollapsedValues/getUIFilterFormUncollapsedValues';

import { updateUIFilterForm } from 'services/filterForm/updateUIFilterForm/updateUIFilterForm';
import { changeSqlCodeVisibility } from 'services/sqlCode/sqlCode';

import { LogsColumns } from 'components/LogsTable/LogsColumns/LogsColumns';

import './LogsToolbar.css';

const cls = cn('logs-toolbar');

interface LogsToolbarProps {
    className?: string;
}

export const LogsToolbar = function LogsToolbar(props: LogsToolbarProps) {
    const dispatch = useDispatch();
    const sql = useSelector(getLogsSql);
    const downloadIcon = useMemo(() => <DownloadOutlined />, []);
    const selectIcon = useMemo(() => <SelectOutlined />, []);
    const uncollapsedValues = useSelector(getUIFilterFormUncollapsedValues);
    const savedForm = useSelector(getSavedForm);

    const onShowSqlClick = useCallback(
        (e) => {
            dispatch(changeSqlCodeVisibility({ isVisible: e.target.checked }));
        },
        [dispatch],
    );

    const onChangeUnCollapsedValues = useCallback(
        (e) => {
            dispatch(
                updateUIFilterForm({
                    uncollapsedValues: e.target.checked,
                }),
            );
        },
        [dispatch],
    );

    return (
        <div className={cls(null, [props.className])}>
            <Button href={getDownloadLogsUrl(savedForm)} type="link" className={cls('button')} icon={downloadIcon}>
                Download
            </Button>
            <Checkbox className={cls('button')} onChange={onShowSqlClick}>
                SQL
            </Checkbox>

            {sql && (
                <Button type="link" className={cls('button')} icon={selectIcon} href={getYqlUrl(sql)} target={'_blank'}>
                    Open in YQL
                </Button>
            )}

            <Checkbox onChange={onChangeUnCollapsedValues} className={cls('button')} checked={uncollapsedValues}>
                uncollapsed values
            </Checkbox>

            <LogsColumns />
        </div>
    );
};
