import * as React from 'react';
import { useMemo } from 'react';
import { useSelector } from 'react-redux';
import { SearchOutlined } from '@ant-design/icons';
import { Dropdown, Menu } from 'antd';

import { FormSchema } from 'schema/form/FormSchema';
import { LogsConfigSchema } from 'schema/logs/LogsConfigSchema';

import { cn } from 'helpers/className/className';
import { getFromDateForApi } from 'helpers/date/getFromDateForApi/getFromDateForApi';
import { getToDateForApi } from 'helpers/date/getToDateForApi/getToDateForApi';
import { getDefaultFieldsFromLogConfig } from 'helpers/form/getDefaultFieldsFromLogConfig/getDefaultFieldsFromLogConfig';
import { formatUrlHash } from 'helpers/url/formatUrlHash/formatUrlHash';

import { getLogsConfig } from 'selectors/logs/getLogsConfig/getLogsConfig';
import { getUIFilterFormFrom } from 'selectors/ui/getUIFilterFormFrom/getUIFilterFormFrom';
import { getUIFilterFormLimit } from 'selectors/ui/getUIFilterFormLimit/getUIFilterFormLimit';
import { getUIFilterFormTo } from 'selectors/ui/getUIFilterFormTo/getUIFilterFormTo';

interface QuickSearchItemsProps {
    className?: string;
    column: 'reqid' | 'trace_id';
    value: string;
}

const cls = cn('quick-search-items');

interface QuickSearchItemParams {
    title: string;
    href: string;
}

interface FormParams {
    from: string;
    to: string;
    limit: number;
}

function getQuickItemsParams(
    logsConfig: Optional<LogsConfigSchema>,
    column: 'reqid' | 'trace_id',
    value: string,
    formParams: FormParams,
): QuickSearchItemParams[] {
    if (!logsConfig) {
        return [];
    }

    // Throw away queryId param
    let hrefWithoutParams = location.pathname;

    return Object.keys(logsConfig).reduce((params, logSource): QuickSearchItemParams[] => {
        const logConfig = logsConfig[logSource];

        let idColumn;
        if (column === 'reqid') {
            idColumn = logConfig.reqidColumn;
        } else {
            idColumn = logConfig.traceIdColumn;
        }

        if (idColumn === undefined || idColumn === '') {
            return params;
        }

        let conditions = {
            [idColumn]: value,
        };
        let fields = getDefaultFieldsFromLogConfig(logConfig);
        let formInput: FormSchema = {
            logType: logSource,
            form: {
                from: formParams.from,
                to: formParams.to,
                fields,
                conditions,
                limit: formParams.limit,
                offset: 0,
                showTraceIdRelated: false,
                reverseOrder: false,
            },
        };

        params.push({
            title: logSource,
            href: hrefWithoutParams + '#' + formatUrlHash(formInput),
        });

        return params;
    }, [] as QuickSearchItemParams[]);
}

const trigger: Array<'click'> = ['click'];

export const QuickSearchItems = React.memo(({ className, column, value }: QuickSearchItemsProps) => {
    const logsConfig = useSelector(getLogsConfig);
    const from = useSelector(getUIFilterFormFrom);
    const to = useSelector(getUIFilterFormTo);
    const limit = useSelector(getUIFilterFormLimit);
    const menuItems = useMemo(
        () =>
            getQuickItemsParams(logsConfig, column, value, {
                from: getFromDateForApi(from),
                to: getToDateForApi(to),
                limit: limit || 100,
            }),
        [column, from, limit, logsConfig, to, value],
    );
    const menu = useMemo(() => {
        return (
            <Menu>
                {menuItems.map((item) => (
                    <Menu.Item key={item.title}>
                        <a target="_blank" rel="noopener noreferrer" href={item.href}>
                            {item.title}
                        </a>
                    </Menu.Item>
                ))}
            </Menu>
        );
    }, [menuItems]);

    return (
        <Dropdown trigger={trigger} className={cls(null, [className])} overlay={menu}>
            <SearchOutlined />
        </Dropdown>
    );
});
