import * as React from 'react';
import { useCallback, useEffect } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { HistoryOutlined, HomeOutlined, QuestionCircleOutlined } from '@ant-design/icons';
import { Layout, Menu } from 'antd';

import { UIPanelType } from 'schema/ui/UISchema';

import { cn } from 'helpers/className/className';

import {getFetchLogsFinishTime} from 'selectors/logs/getFetchLogsFinishTime/getFetchLogsFinishTime';
import { getUISelectedPanel } from 'selectors/ui/getUISelectedPanel/getUISelectedPanel';

import { fulfillForm } from 'services/fulfillForm/fulfillForm';
import { switchUIPanel } from 'services/panels/switchUIPanel';

import { HistoryPanel } from 'components/HistoryPanel/HistoryPanel';
import { Logs } from 'components/Logs/Logs';

import './Page.css';
import { isLogsDataLoading } from 'selectors/logs/isLogsDataLoading/isLogsDataLoading';
import { isPageDataLoadingByQueryId } from 'selectors/logs/isPageDataLoadingByQueryId/isPageDataLoadingByQueryId';

const { Content, Sider } = Layout;

const cls = cn('page');

export const Page = function Page() {
    let dispatch = useDispatch();
    const abortControllerRef = React.useRef(new AbortController());
    const fetchLogsFinishTime = useSelector(getFetchLogsFinishTime);
    const abortFetchLogsRequest = useCallback(() => {
        abortControllerRef.current.abort();
    }, []);

    const getFetchLogsRequestSignal = useCallback(() => {
        return abortControllerRef.current.signal;
    }, []);

    useEffect(() => {
        dispatch(fulfillForm({ signal: getFetchLogsRequestSignal() }));
    }, [dispatch, getFetchLogsRequestSignal]);

    useEffect(() => {
        if (fetchLogsFinishTime !== undefined) {
            abortControllerRef.current = new AbortController();
        }
    }, [fetchLogsFinishTime]);

    const onMenuItemSelect = useCallback(
        ({ key }) => {
            dispatch(switchUIPanel({ selectedPanel: key }));
        },
        [dispatch],
    );

    const selectedPanel = useSelector(getUISelectedPanel);
    const logsDataLoading = useSelector(isLogsDataLoading);
    const isPageDataLoadingByQueryIdValue = useSelector(isPageDataLoadingByQueryId);
    const showHistory = selectedPanel === UIPanelType.History;
    const menuWidth = 80;
    const historyWidth = 200;
    const contentMargin = showHistory ? menuWidth + historyWidth : menuWidth;

    return (
        <div className={cls()}>
            <Layout className={cls('layout')}>
                <Sider
                    width={menuWidth}
                    collapsible={false}
                    collapsed
                    style={{
                        overflow: 'auto',
                        height: '100vh',
                        position: 'fixed',
                        left: 0,
                    }}
                >
                    <Menu
                        theme="dark"
                        selectedKeys={[selectedPanel]}
                        defaultSelectedKeys={[UIPanelType.Home]}
                        mode="inline"
                        onSelect={onMenuItemSelect}
                    >
                        <Menu.Item key={UIPanelType.Home} icon={<HomeOutlined />} />
                        <Menu.Item disabled={logsDataLoading || isPageDataLoadingByQueryIdValue} key={UIPanelType.History} icon={<HistoryOutlined />}>
                            History
                        </Menu.Item>
                        <Menu.Item key={UIPanelType.Documentation} icon={<QuestionCircleOutlined />}>
                            <a
                                href={'https://docs.yandex-team.ru/direct-dev/reference/logviewer/quick-start'}
                                target="_blank"
                                rel="noopener noreferrer"
                            >
                                Documentation
                            </a>
                        </Menu.Item>
                    </Menu>
                </Sider>
                {showHistory && (
                    <Sider
                        width={historyWidth}
                        theme={'light'}
                        collapsible={false}
                        style={{
                            overflow: 'auto',
                            height: '100vh',
                            position: 'fixed',
                            left: menuWidth,
                        }}
                    >
                        <HistoryPanel />
                    </Sider>
                )}
                <Layout className="site-layout" style={{ marginLeft: contentMargin }}>
                    <Content className={cls('content')}>
                        <div className={cls('old-interface-link-block')}>
                            <a href="/logviewer/switchUi?new=false" target="_blank" className={cls('old-interface-link')}>Old Interface</a>
                        </div>
                        <Logs
                            abortFetchLogsRequest={abortFetchLogsRequest}
                            getFetchLogsRequestSignal={getFetchLogsRequestSignal}
                        />
                    </Content>
                </Layout>
            </Layout>
        </div>
    );
};
