import * as React from 'react';
import { useCallback, useEffect, useRef } from 'react';
import { useSelector } from 'react-redux';
import { CopyOutlined } from '@ant-design/icons';
import { Button, Card } from 'antd';
import hljs from 'highlight.js';

import { cn } from 'helpers/className/className';
import { copyToClipboard } from 'helpers/common/copyToClipboard/copyToClipboard';
import { showMessage } from 'helpers/notification/showMessage/showMessage';

import { getLogsSql } from 'selectors/logs/getLogsSql/getLogsSql';
import { getUIIsSqlBlockVisible } from 'selectors/ui/getUIIsSqlBlockVisible/getUIIsSqlBlockVisible';

import 'highlight.js/styles/github.css';
import './SqlCode.css';

interface SqlCodeProps {
    className: string;
}

const cls = cn('sql-code');

export const SqlCode = React.memo((props: SqlCodeProps) => {
    const preRef = useRef<HTMLPreElement>(null);
    const sql = useSelector(getLogsSql);
    const isSqlCodeVisible = useSelector(getUIIsSqlBlockVisible);
    const { className } = props;

    useEffect(() => {
        if (preRef && preRef.current) {
            hljs.highlightBlock(preRef.current);
        }
    }, [preRef]);

    const onCopyClick = useCallback(() => {
        copyToClipboard(sql || '').then(() => {
            showMessage('Copied to clipboard', 'success');
        });
    }, [sql]);

    return (
        <Card hidden={!isSqlCodeVisible} className={cls(null, [className])}>
            <pre ref={preRef}>
                <code>{sql}</code>
            </pre>

            <Button
                title={'Copy value'}
                onClick={onCopyClick}
                className={cls('copy-button')}
                icon={<CopyOutlined />}
                size={'small'}
            >
                Copy
            </Button>
        </Card>
    );
});
