import moment, { Moment, unitOfTime } from 'moment';

import { FormSchema } from 'schema/form/FormSchema';

import { getDefaultFrom } from 'helpers/date/getDefaultFrom/getDefaultFrom';
import { getDefaultTo } from 'helpers/date/getDefaultTo/getDefaultTo';

import { DATE_SHOW_LOGS_API_FORMAT } from 'enums/date';

export function getSavedFormFromHash(hashState: Partial<FormSchema>): FormSchema {
    let logType = hashState.logType;
    let form = hashState.form;

    let from: Moment;

    if (!form || form.from === undefined) {
        from = getDefaultFrom();
    } else if (form.from.startsWith('now')) {
        // e.g. now-1h
        const relTimePattern = /now(-\d+)([dhm])/g;
        const match = relTimePattern.exec(form.from);
        // default values
        let num = -1;
        let unit: unitOfTime.DurationConstructor = 'hours';

        if (match != null) {
            num = Number(match[1]);
            switch (match[2]) {
                case 'd':
                    unit = 'days';
                    break;
                case 'h':
                    unit = 'hours';
                    break;
                case 'm':
                    unit = 'minutes';
                    break;
            }
        }

        from = moment().add(num, unit);
    } else {
        from = moment(form.from);
    }

    let to;
    if (!form || form.to === undefined) {
        to = getDefaultTo();
    } else {
        to = moment(form.to);
    }

    if (to.valueOf() < from.valueOf()) {
        to = from.endOf('day');
    }

    let savedForm: FormSchema = {
        logType: logType || '',
        form: {
            conditions: {},
            fields: [],
            from: from.format(DATE_SHOW_LOGS_API_FORMAT),
            limit: (form && form.limit) || 100,
            offset: (form && form.offset) || 0,
            reverseOrder: Boolean(form && form.reverseOrder),
            showTraceIdRelated: Boolean(form && form.showTraceIdRelated),
            to: to.format(DATE_SHOW_LOGS_API_FORMAT),
        },
    };

    if (form && form.showStats) {
        savedForm.form.showStats = true;
        savedForm.form.logTimeGroupBy = form.logTimeGroupBy;
        savedForm.form.sortByCount = form.sortByCount;
    }

    if (form && Array.isArray(form.fields)) {
        savedForm.form.fields = form.fields;
    }

    if (form && form.conditions) {
        savedForm.form.conditions = form.conditions;
    }

    return savedForm;
}
