import { FetchRequestOptions, RequestInterface } from './RequestInterface';

/* eslint-disable @typescript-eslint/no-explicit-any */


const DEFAULT_HEADERS: Dict<string> = {
    Accept: 'application/json',
};

export class ClientRequest implements RequestInterface {
    public get<T>(url: string, options: FetchRequestOptions = {}): Promise<T> {
        return this.fetch<T>(url, options);
    }

    public post<T>(url: string, data: any, options: FetchRequestOptions = {}): Promise<T> {
        return this._post<T>(url, data, options);
    }

    private async fetch<T>(url: string, options: FetchRequestOptions = {}): Promise<T> {
        let headers: Dict<string> = options.headers || {};
        let response: Response | undefined;
        let responseJSON: any;

        let opts: RequestInit = {
            headers: {
                ...DEFAULT_HEADERS,
                ...headers,
            },
            credentials: 'same-origin',
            ...options.options,
        };

        let fetchUrl = url;

        try {
            response = await fetch(fetchUrl, opts);

            try {
                if (options.isText) {
                    responseJSON = await response.text();
                } else {
                    responseJSON = await response.json();
                }
            } catch (error) {
                if (error.name === 'SyntaxError' || error.name === 'AbortError') {
                    responseJSON = null;
                } else {
                    throw error;
                }
            }

            if (response.status >= 400) {
                throw responseJSON;
            }

            return responseJSON;
        } catch (error) {
            error.status = 0;
            error.response = response;
            error.responseData = responseJSON;

            throw error;
        }
    }

    private async _post<T>(url: string, data: any, options: FetchRequestOptions = {}): Promise<T> {
        let isFormData = data instanceof FormData;

        return this.fetch<T>(url, {
            ...options,
            headers: isFormData
                ? options.headers
                : {
                      ...options.headers,
                      'content-type': 'application/json; charset=utf-8',
                  },
            options: {
                method: 'POST',
                cache: 'no-cache',
                redirect: 'error',
                body: isFormData ? data : JSON.stringify(data),

                ...options.options,
            },
        });
    }
}
