import moment from 'moment';

import { createService } from 'state/createService/createService';

import { FormSchema } from 'schema/form/FormSchema';

import { mergeState } from 'mutators/mergeState/mergeState';
import { mergeStateUIFilterForm } from 'mutators/mergeStateUIFilterForm/mergeStateUIFilterForm';

import { getSavedFormFromHash } from 'helpers/form/getSavedFormFromHash/getSavedFormFromHash';
import { getSavedFormFromOldFormatHash } from 'helpers/form/getSavedFormFromOldFormatHash/getSavedFormFromOldFormatHash';

import { fetchLogs } from 'services/fetchLogs/fetchLogs';
import { fetchLogsByQueryId } from 'services/fetchLogsByQueryId/fetchLogsByQueryId';
import { loadConfig } from 'services/loadConfig/loadConfig';

import { DATE_SHOW_LOGS_API_FORMAT } from 'enums/date';

const JSURL = require('jsurl');

function getDefaultFrom() {
    return moment().startOf('day');
}

function getDefaultTo() {
    return moment().endOf('day');
}

export const fulfillForm = createService(async function fulfillForm(
    { signal }:{ signal?: AbortSignal },
    { setState, dispatch, getState }
) {
    await dispatch(loadConfig());
    const urlObj = new URL(location.href);
    const queryId = urlObj.searchParams.get('queryId');

    if (queryId) {
        return dispatch(fetchLogsByQueryId({ queryId }));
    }

    if (!location.hash) {
        let state = mergeStateUIFilterForm(getState(), {
            from: getDefaultFrom().format(DATE_SHOW_LOGS_API_FORMAT),
            to: getDefaultTo().format(DATE_SHOW_LOGS_API_FORMAT),
            conditions: {},
            limit: 100,
            reverseOrder: false,
            showTraceIdRelated: false,
        });

        state = mergeState(state, {
            logs: undefined,
        });

        setState(state);

        return;
    }

    setState(
        mergeState(getState(), {
            logs: {
                _loading: true,
                totalCount: 0,
                data: [],
                sql: '',
                reqid: 0,
            },
        }),
    );

    let hash = location.hash.substr(1);
    const isJson = hash.startsWith(encodeURIComponent('{'));
    const isJSURI = hash.startsWith('~');
    let savedForm: Optional<FormSchema>;

    if (isJson || isJSURI) {
        if (isJSURI && hash.endsWith('$')) {
            hash = hash.substring(0, hash.length - 1);
        }

        let state = isJSURI ? JSURL.parse(hash) : JSON.parse(decodeURIComponent(hash));
        savedForm = getSavedFormFromHash(state);
    } else {
        // Old format
        savedForm = getSavedFormFromOldFormatHash(hash);
    }

    if (!savedForm) {
        return;
    }

    setState(
        mergeStateUIFilterForm(getState(), {
            logType: savedForm.logType,
            ...savedForm.form,
        }),
    );

    dispatch(fetchLogs({ input: savedForm, signal }));
});
