package ru.yandex.direct.logviewer.auth;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.tracing.util.TraceCommentVarsHolder;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieCredentials;
import ru.yandex.direct.web.auth.blackbox.exception.BadBlackboxCredentialsException;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxBadCredentialsException;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxCredentialsExpiredException;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.BAD_CREDENTIALS;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.CREDENTIALS_EXPIRED;
import static ru.yandex.direct.web.core.security.authentication.WebAuthTranslations.WEB_AUTH_TRANSLATIONS;

public class LogviewerAuthFilter implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(LogviewerAuthFilter.class);

    private final AuthenticationManager authenticationManager;
    private final TvmIntegration tvmIntegration;
    private final TvmService externalBlackboxTvmService;
    private final TvmService internalBlackboxTvmService;

    public LogviewerAuthFilter(AuthenticationManager authenticationManager,
                               TvmIntegration tvmIntegration,
                               TvmService externalBlackboxTvmService,
                               TvmService internalBlackboxTvmService) {
        checkNotNull(authenticationManager, "authenticationManager is required");
        this.authenticationManager = authenticationManager;
        this.tvmIntegration = tvmIntegration;
        this.externalBlackboxTvmService = externalBlackboxTvmService;
        this.internalBlackboxTvmService = internalBlackboxTvmService;
    }

    @Override
    public void init(FilterConfig filterConfig) {
    }

    @Override
    public void destroy() {
    }

    private BlackboxType getBlackboxTypeForRequest(HttpServletRequest httpRequest) {
        String requestURL = httpRequest.getRequestURL().toString();
        try {
            String host = new URI(requestURL).getHost();
            if (host.endsWith("yandex-team.ru")) {
                return BlackboxType.INTERNAL;
            } else {
                return BlackboxType.EXTERNAL;
            }
        } catch (URISyntaxException e) {
            logger.warn("Can't create URI from string {}", requestURL, e);
            return BlackboxType.EXTERNAL;
        }
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        Authentication auth;
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        logger.debug("authenticating incoming request: {}", httpRequest.getRequestURL().toString());
        BlackboxCookieCredentials credentials = BlackboxCookieCredentials.extract(httpRequest);

        BlackboxType blackboxType = getBlackboxTypeForRequest(httpRequest);

        String tvmTicket;
        if (blackboxType == BlackboxType.EXTERNAL) {
            tvmTicket = tvmIntegration.getTicket(externalBlackboxTvmService);
        } else {
            tvmTicket = tvmIntegration.getTicket(internalBlackboxTvmService);
        }

        LogviewerCookieAuthRequest authRequest = new LogviewerCookieAuthRequest(
                credentials, tvmTicket, blackboxType);

        try {
            auth = authenticationManager.authenticate(authRequest);
        } catch (BadCredentialsException | BadBlackboxCredentialsException e) {
            throw new BlackboxBadCredentialsException("Bad credentials", e, BAD_CREDENTIALS,
                    WEB_AUTH_TRANSLATIONS.badCredentials(), null);
        } catch (CredentialsExpiredException e) {
            throw new BlackboxCredentialsExpiredException("Credentials has expired", e, CREDENTIALS_EXPIRED,
                    WEB_AUTH_TRANSLATIONS.credentialsHasExpired(), null);
        }

        if (auth != null) {
            SecurityContextHolder.getContext().setAuthentication(auth);
        }

        try {
            if (auth instanceof DirectAuthentication) {
                DirectAuthentication directAuth = (DirectAuthentication) auth;
                if (directAuth.getOperator() != null && directAuth.getOperator().getUid() != null) {
                    TraceCommentVarsHolder.get().setOperator(String.valueOf(directAuth.getOperator().getUid()));
                }
            }
            chain.doFilter(request, response);
        } finally {
            TraceCommentVarsHolder.get().removeOperator();
        }
    }
}
