package ru.yandex.direct.logviewer.auth;

import java.util.List;

import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.authority.SimpleGrantedAuthority;

import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuth;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthProvider;
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * AuthenticationManager, который умеет пускать пользователей и внутреннего (yandex-team), и внешнего (yandex.ru)
 * паспортов. В случае использования внешнего паспорта пользователь авторизуется через Директ и получает
 * роли из базы Директа. Если же используется внутренний паспорт, то роли будут браться из IDM для системы Logviewer.
 */
public class LogviewerAuthenticationManager implements AuthenticationManager {
    private final BlackboxCookieAuthProvider externalBlackboxCookieAuthProvider;
    private final BlackboxCookieAuthProvider internalBlackboxCookieAuthProvider;
    private final DirectCookieAuthProvider directCookieAuthProvider;
    private final LogviewerUserRolesService logviewerUserRolesService;

    public LogviewerAuthenticationManager(
            BlackboxCookieAuthProvider externalBlackboxCookieAuthProvider,
            BlackboxCookieAuthProvider internalBlackboxCookieAuthProvider,
            DirectCookieAuthProvider directCookieAuthProvider,
            LogviewerUserRolesService logviewerUserRolesService
    ) {
        this.externalBlackboxCookieAuthProvider = externalBlackboxCookieAuthProvider;
        this.internalBlackboxCookieAuthProvider = internalBlackboxCookieAuthProvider;
        this.directCookieAuthProvider = directCookieAuthProvider;
        this.logviewerUserRolesService = logviewerUserRolesService;
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        LogviewerCookieAuthRequest authenticationReq = (LogviewerCookieAuthRequest) authentication;
        BlackboxCookieAuth authenticate;
        if (authenticationReq.getBlackboxType() == BlackboxType.INTERNAL) {
            // Если пользователь логинился через yandex-team, то его роли получаем из YDB
            authenticate = internalBlackboxCookieAuthProvider.authenticate(authenticationReq);
            List<String> userRoles = logviewerUserRolesService.getUserRoles(authenticate.getPrincipal());
            List<SimpleGrantedAuthority> authorities = mapList(userRoles, SimpleGrantedAuthority::new);
            return new LogviewerInternalBlackboxAuthentication(authenticate.getPrincipal(), authorities);
        } else {
            // Если пользователь логинился через direct.yandex.ru, то берём его доступы из базы Директа
            authenticate = externalBlackboxCookieAuthProvider.authenticate(authenticationReq);
            return directCookieAuthProvider.authenticate(authenticate);
        }
    }
}
