package ru.yandex.direct.logviewer.auth

import com.google.common.base.Preconditions
import com.google.common.collect.Sets
import com.yandex.ydb.table.values.PrimitiveType
import one.util.streamex.EntryStream
import org.springframework.security.core.context.SecurityContextHolder
import ru.yandex.direct.core.security.DirectAuthentication
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.logviewercore.domain.ppclog.LogTable
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.column.Column
import ru.yandex.direct.ydb.table.Table
import java.util.*
import java.util.stream.Collectors.toSet

/**
 * Descriptor class for the YDB table
 */
class LogviewerUserRolesTable : Table("logviewer_user_roles") {
    val LOGIN = Column<String>(this, "login", PrimitiveType.utf8()).pk()
    val ROLE = Column<String>(this, "role", PrimitiveType.utf8()).pk()

    private val COLUMNS = listOf(LOGIN, ROLE)

    override fun getColumns() = COLUMNS
}

class LogviewerUserRolesService(
    val ydbClient: YdbClient,
    val path: YdbPath,
    private val environmentType: EnvironmentType,
) {
    companion object {
        private val USER_ROLES = LogviewerUserRolesTable()

        const val ROLE_LOGVIEWER_DIRECT = "ROLE_LOGVIEWER_DIRECT"
        const val ROLE_LOGVIEWER_GRUT = "ROLE_LOGVIEWER_GRUT"

        const val IDM_ROLE_SLUG = "log_viewer_scope"

        const val IDM_ROLE_DIRECT = "direct"
        const val IDM_ROLE_GRUT = "grut"
    }

    /**
     * Таблица соответствия между ролью в IDM и ролью в spring security
     */
    private val IDM_ROLES_TO_ROLE = mapOf(
        IDM_ROLE_DIRECT to ROLE_LOGVIEWER_DIRECT,
        IDM_ROLE_GRUT to ROLE_LOGVIEWER_GRUT
    )

    /**
     * Список ролей в Директе, которые дают доступ в логвьювер при заходе через direct.yandex.ru
     */
    private val DIRECT_ROLES = setOf(
        "ROLE_SUPER",
        "ROLE_SUPERREADER",
        "ROLE_SUPPORT",
        "ROLE_LIMITED_SUPPORT",
        "ROLE_INTERNAL_AD_ADMIN",
        "ROLE_MANAGER",
        "ROLE_PLACER"
    )

    /**
     * Логи Директа могут просматривать пользователи Директа, имеющие одну из ролей DIRECT_ROLES,
     * а также коллеги, авторизованные во внутреннем паспорте с ролью ROLE_LOGVIEWER_DIRECT
     * (это уже отдельная роль для системы Logviewer)
     *
     *
     * Логи Грута могут просматривать пользователи Директа с ролью SUPERREADER,
     * а также те, кто авторизуется во внутреннем паспорте и имеет роль ROLE_LOGVIEWER_GRUT
     * в системе Logviewer
     */
    private val SCOPE_VIEWERS_ROLES = mapOf(
        LogTable.SCOPE_DIRECT to Sets.union(DIRECT_ROLES, setOf(ROLE_LOGVIEWER_DIRECT)),
        LogTable.SCOPE_GRUT to setOf("ROLE_SUPERREADER", ROLE_LOGVIEWER_GRUT)
    )

    /**
     * Обратная мапа, чтобы по ролям пользователя определять набор скоупов, к которым он имеет доступ
     */
    private val ROLE_TO_SCOPE = EntryStream.of(SCOPE_VIEWERS_ROLES)
        .flatMapValues { obj -> obj.stream() }
        .invert()
        .grouping({ HashMap() }, toSet())

    fun getCurrentUserLogin(): String {
        val authentication = SecurityContextHolder.getContext().authentication
        Preconditions.checkState(
            authentication is DirectAuthentication
                || authentication is LogviewerInternalBlackboxAuthentication
        )
        return authentication.name
    }

    fun getCurrentUserRoles(): Set<String> {
        val authentication = SecurityContextHolder.getContext().authentication
        Preconditions.checkState(
            (authentication is DirectAuthentication
                || authentication is LogviewerInternalBlackboxAuthentication)
        )
        return authentication.authorities
            .map { it.authority }
            .toSet()
    }

    fun getCurrentUserScopes(): Set<String> {
        val roles = getCurrentUserRoles()
        return roles.mapNotNull { ROLE_TO_SCOPE[it] }
            .flatten()
            .toSet()
    }

    fun hasAccessToLog(userScopes: Set<String>, allowAccessFor: Array<String>): Boolean {
        return Arrays.stream(allowAccessFor).anyMatch { userScopes.contains(it) }
    }

    fun getUserRoles(login: String): List<String> {
        // На непродакшеновых средах слишком сложно синхронизировать роли с продакшеном,
        // а раздавать роли отдельно для тестинга не хочется.
        // Проще давать доступ всем, у кого есть сетевые доступы
        if (!environmentType.isProduction) {
            return listOf(ROLE_LOGVIEWER_DIRECT, ROLE_LOGVIEWER_GRUT)
        }
        val queryAndParams = SelectBuilder.select(
            USER_ROLES.LOGIN,
            USER_ROLES.ROLE
        )
            .from(USER_ROLES)
            .where(USER_ROLES.LOGIN.eq(login))
            .queryAndParams(path)
        val reader = ydbClient.executeQuery(queryAndParams).getResultSet(0)

        val idmRoles = mutableListOf<String>()
        while (reader.next()) {
            idmRoles.add(reader.getValueReader(USER_ROLES.ROLE).utf8)
        }
        return idmRoles.mapNotNull(IDM_ROLES_TO_ROLE::get)
    }
}
