package ru.yandex.direct.logviewer.configuration;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Objects;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import com.google.common.net.MediaType;
import one.util.streamex.StreamEx;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxBadCredentialsException;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxCookieAuthenticationException;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxCredentialsExpiredException;

import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.web.auth.blackbox.PassportUrls.AUTH;
import static ru.yandex.direct.web.auth.blackbox.PassportUrls.AUTH_UPDATE;

/**
 * Исключения SpringSecurity нельзя отловить ControllerAdvice-ом, поэтому
 * ловим отдельным фильтром. В случае ajax запросов возвращается код 402
 * (он плохо подходит, но все остальные подходят хуже) и json с информацией о ошибке
 * и урлом паспорта для редиректа. В случае прочих запросов отдаём честный редирект.
 */
@Component
public class AuthExceptionFilter extends OncePerRequestFilter {
    private final TranslationService translationService;

    public AuthExceptionFilter(TranslationService translationService) {
        this.translationService = translationService;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {
        try {
            filterChain.doFilter(request, response);
        } catch (BlackboxCredentialsExpiredException e) {
            sendAuthUpdate(request, response);
        } catch (BlackboxBadCredentialsException e) {
            sendNeedAuth(request, response);
        } catch (BlackboxCookieAuthenticationException e) {
            sendForbidden(request, response, e);
        }
    }

    private void sendForbidden(HttpServletRequest request, HttpServletResponse response,
                               BlackboxCookieAuthenticationException e) throws IOException {
        response.reset();
        String message = StreamEx.of(e.getShortMessage(), e.getDetailedMessage())
                .filter(Objects::nonNull)
                .map(translationService::translate)
                .joining(", ");
        if (HttpUtil.isAjax(request)) {
            sendResponse(response, HttpStatus.FORBIDDEN, MediaType.JSON_UTF_8,
                    toJson(new AjaxAuthError("forbidden", message, null))
            );
        } else {
            sendResponse(response, HttpStatus.FORBIDDEN, MediaType.PLAIN_TEXT_UTF_8,
                    "forbidden: " + message
            );
        }
    }

    private void sendAuthUpdate(HttpServletRequest request, HttpServletResponse response) throws IOException {
        response.reset();
        if (HttpUtil.isAjax(request)) {
            sendResponse(response, HttpStatus.PAYMENT_REQUIRED, MediaType.JSON_UTF_8,
                    toJson(new AjaxAuthError("need_reset", "credentials expired",
                            AUTH_UPDATE.buildWithoutRetpath(request)))
            );
        } else {
            response.sendRedirect(AUTH_UPDATE.build(request));
        }
    }

    private void sendNeedAuth(HttpServletRequest request, HttpServletResponse response) throws IOException {
        response.reset();
        if (HttpUtil.isAjax(request)) {
            sendResponse(response, HttpStatus.PAYMENT_REQUIRED, MediaType.JSON_UTF_8,
                    toJson(new AjaxAuthError("need_auth", "no authorization", AUTH.buildWithoutRetpath(request)))
            );
        } else {
            response.sendRedirect(AUTH.build(request));
        }
    }

    private void sendResponse(HttpServletResponse response, HttpStatus status, MediaType type, String text)
            throws IOException {
        response.setStatus(status.value());
        if (type.charset().isPresent()) {
            response.setCharacterEncoding(type.charset().get().name());
        }
        response.setHeader(HttpHeaders.CONTENT_TYPE, type.toString());
        PrintWriter writer = response.getWriter();
        writer.write(text);
        writer.flush();
    }

    /**
     * Данные о ошибке авторизации, отправляемые в браузер в случае ajax-запроса
     */
    @JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY)
    @JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
    public static class AjaxAuthError {
        private final String reason;
        private final String details;
        private final String urlBase;

        public AjaxAuthError(String reason, String details, String urlBase) {
            this.reason = reason;
            this.details = details;
            this.urlBase = urlBase;
        }

        public String getReason() {
            return reason;
        }

        public String getDetails() {
            return details;
        }

        public String getUrlBase() {
            return urlBase;
        }
    }
}
