package ru.yandex.direct.logviewer.configuration;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.web.DefaultSecurityFilterChain;
import org.springframework.security.web.util.matcher.AndRequestMatcher;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.NegatedRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.integrations.configuration.IntegrationsConfiguration;
import ru.yandex.direct.integrations.configuration.IntegrationsUtils;
import ru.yandex.direct.logviewer.auth.LogviewerAuthFilter;
import ru.yandex.direct.logviewer.auth.LogviewerAuthenticationManager;
import ru.yandex.direct.logviewer.auth.LogviewerUserRolesService;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthProvider;
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider;
import ru.yandex.direct.web.core.security.authentication.WebAuthenticationFilter;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.client.YdbClient;

import static ru.yandex.direct.logviewer.configuration.LogviewerYdbConfiguration.LOGVIEWER_YDB_CLIENT_BEAN;
import static ru.yandex.direct.logviewer.configuration.LogviewerYdbConfiguration.LOGVIEWER_YDB_PATH_BEAN;

@Configuration
@Import({
        CoreConfiguration.class,
        IntegrationsConfiguration.class,
        LogviewerYdbConfiguration.class
})
public class LogviewerAuthenticationConfiguration {

    public static final String LOGVIEWER_AUTHENTICATION_FILTER = "logviewerAuthenticationFilter";

    private static final String MATCH_ALL_URLS = "/**";
    private static final String MATCH_ALIVE_URL = "/alive";
    private static final String MATCH_ADMIN_URL = "/admin";
    private static final String MATCH_IDM_URL = "/idm/**";
    private static final String MATCH_MONITORING_URL = "/monitoring";

    private static final int MAX_BLACKBOX_CONNECTIONS = 100;

    @Bean(name = LOGVIEWER_AUTHENTICATION_FILTER)
    public WebAuthenticationFilter blackBoxWebAuthenticationFilter(LogviewerAuthFilter logviewerAuthFilter) {
        RequestMatcher logviewerRequestMatcher = new AndRequestMatcher(
                new AntPathRequestMatcher(MATCH_ALL_URLS),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_ALIVE_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_ADMIN_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_MONITORING_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_IDM_URL))
        );
        DefaultSecurityFilterChain logviewerAuthSecurityChain =
                new DefaultSecurityFilterChain(logviewerRequestMatcher, logviewerAuthFilter);

        return new WebAuthenticationFilter(logviewerAuthSecurityChain);
    }

    @Bean
    public LogviewerUserRolesService logviewerUsersService(
            @Qualifier(LOGVIEWER_YDB_CLIENT_BEAN) YdbClient ydbClient,
            @Qualifier(LOGVIEWER_YDB_PATH_BEAN) YdbPath path,
            EnvironmentType environmentType
    ) {
        return new LogviewerUserRolesService(ydbClient, path, environmentType);
    }

    @Bean
    public LogviewerAuthFilter logviewerAuthFilter(DirectConfig directConfig,
                                                   UserService userService, RbacService rbacService,
                                                   NetAcl netAcl, ClientService clientService,
                                                   TvmIntegration tvmIntegration,
                                                   EnvironmentType environmentType,
                                                   LogviewerUserRolesService logviewerUsersService) {
        TvmService externalBlockboxTvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
        TvmService internalBlackboxTvmService = TvmService.BLACKBOX_PROD_YATEAM;

        return new LogviewerAuthFilter(
                logviewerAuthManager(directConfig, userService, rbacService, netAcl, clientService,
                        logviewerUsersService),
                tvmIntegration, externalBlockboxTvmService, internalBlackboxTvmService);
    }

    private LogviewerAuthenticationManager logviewerAuthManager(DirectConfig directConfig,
                                                                UserService userService, RbacService rbacService,
                                                                NetAcl netAcl, ClientService clientService,
                                                                LogviewerUserRolesService logviewerUserRolesService) {
        BlackboxClient externalBlackboxClient = IntegrationsUtils.configureBlackboxClient(
                directConfig, "blackbox", MAX_BLACKBOX_CONNECTIONS);
        BlackboxClient internalBlackboxClient = IntegrationsUtils.configureBlackboxClient(
                directConfig, "blackbox_yandex_team", MAX_BLACKBOX_CONNECTIONS);
        return new LogviewerAuthenticationManager(
                blackboxCookieAuthProvider(externalBlackboxClient),
                blackboxCookieAuthProvider(internalBlackboxClient),
                directCookieAuthProvider(userService, rbacService, netAcl, clientService),
                logviewerUserRolesService
        );
    }

    private BlackboxCookieAuthProvider blackboxCookieAuthProvider(BlackboxClient blackboxClient) {
        return new BlackboxCookieAuthProvider(blackboxClient);
    }

    private DirectCookieAuthProvider directCookieAuthProvider(UserService userService,
                                                              RbacService rbacService,
                                                              NetAcl netAcl, ClientService clientService) {
        return new DirectCookieAuthProvider(userService, rbacService, netAcl, clientService);
    }
}
