package ru.yandex.direct.logviewer.configuration;

import java.time.Duration;

import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.auth.TokenAuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.common.configuration.TvmIntegrationConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.client.YdbSessionProperties;
import ru.yandex.direct.ydb.tvm.YdbTvmAuthProvider;

@Lazy
@Configuration
@Import(TvmIntegrationConfiguration.class)
public class LogviewerYdbConfiguration {
    public static final String LOGVIEWER_YDB_AUTH_PROVIDER_BEAN = "LogviewerYdbAuthProviderBean";
    public static final String LOGVIEWER_YDB_PATH_BEAN = "LogviewerYdbPathBean";
    public static final String LOGVIEWER_YDB_RPC_TRANSPORT_BEAN = "LogviewerYdbRpcTransportBean";
    public static final String LOGVIEWER_YDB_TABLE_CLIENT_BEAN = "LogviewerYdbTableClientBean";
    public static final String LOGVIEWER_YDB_SESSION_PROPERTIES_BEAN = "LogviewerYdbSessionPropertiesBean";
    public static final String LOGVIEWER_YDB_CLIENT_BEAN = "LogviewerYdbClientBean";

    @Bean(LOGVIEWER_YDB_AUTH_PROVIDER_BEAN)
    public AuthProvider ydbAuthProvider(DirectConfig directConfig, TvmIntegration tvmIntegration) {
        var tokenPath = directConfig
                .getBranch("maintenance-helpers")
                .getBranch("ydb").findString("token_file");
        if (tokenPath.isPresent()) {
            var path = FileUtils.expandHome(tokenPath.get());
            var token = FileUtils.slurp(path).trim();
            return new TokenAuthProvider(token);
        } else {
            return new YdbTvmAuthProvider(tvmIntegration);
        }
    }

    @Bean(LOGVIEWER_YDB_PATH_BEAN)
    public YdbPath ydbPath(
            @Value("${maintenance-helpers.ydb.cluster}") String cluster,
            @Value("${maintenance-helpers.ydb.db_name}") String dbName
    ) {
        return YdbPath.of(cluster, dbName);
    }

    @Bean(LOGVIEWER_YDB_RPC_TRANSPORT_BEAN)
    public RpcTransport grpcTransport(
            @Value("${maintenance-helpers.ydb.endpoint}") String endpoint,
            @Qualifier(LOGVIEWER_YDB_PATH_BEAN) YdbPath dbPath,
            @Qualifier(LOGVIEWER_YDB_AUTH_PROVIDER_BEAN) AuthProvider authProvider
    ) {
        return GrpcTransport.forEndpoint(endpoint, dbPath.getPath())
                .withAuthProvider(authProvider)
                .build();
    }

    @Bean(LOGVIEWER_YDB_TABLE_CLIENT_BEAN)
    public TableClient tableClient(
            @Qualifier(LOGVIEWER_YDB_RPC_TRANSPORT_BEAN) RpcTransport rpcTransport,
            @Value("${maintenance-helpers.ydb.session_pool_min_size}") int sessionPoolMinSize,
            @Value("${maintenance-helpers.ydb.session_pool_max_size}") int sessionPoolMaxSize
    ) {
        return TableClient.newClient(GrpcTableRpc.useTransport(rpcTransport))
                .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
                .build();
    }

    @Bean(LOGVIEWER_YDB_SESSION_PROPERTIES_BEAN)
    public YdbSessionProperties ydbProperties(
            @Value("${maintenance-helpers.ydb.max_query_retries}") int maxQueryRetries,
            @Value("${maintenance-helpers.ydb.retry_not_found}") Boolean retryNotFound
    ) {
        return YdbSessionProperties.builder()
                .withMaxQueryRetries(maxQueryRetries)
                .withRetryNotFound(retryNotFound)
                .build();
    }

    @Bean(LOGVIEWER_YDB_CLIENT_BEAN)
    public YdbClient ydbClient(
            @Qualifier(LOGVIEWER_YDB_TABLE_CLIENT_BEAN) TableClient tableClient,
            @Qualifier(LOGVIEWER_YDB_SESSION_PROPERTIES_BEAN) YdbSessionProperties ydbSessionProperties,
            @Value("${maintenance-helpers.ydb.query_timeout}") Long queryTimeoutSeconds
    ) {
        var sessionRetryContext = SessionRetryContext.create(tableClient)
                .maxRetries(ydbSessionProperties.getMaxQueryRetries())
                .retryNotFound(ydbSessionProperties.isRetryNotFound())
                .build();
        return new YdbClient(sessionRetryContext, Duration.ofSeconds(queryTimeoutSeconds));
    }
}
