package ru.yandex.direct.logviewer.configuration;

import java.io.IOException;
import java.sql.Timestamp;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;

import javax.servlet.Filter;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.CacheControl;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.security.web.FilterChainProxy;
import org.springframework.web.filter.CharacterEncodingFilter;
import org.springframework.web.filter.CompositeFilter;
import org.springframework.web.servlet.config.annotation.EnableWebMvc;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import ru.yandex.direct.common.tracing.TraceContextFilter;
import ru.yandex.direct.logviewer.logging.LogviewerLoggingFilter;
import ru.yandex.direct.web.core.security.configuration.TvmConfiguration;
import ru.yandex.direct.web.core.security.tvm.TvmInterceptor;

import static ru.yandex.direct.logviewer.configuration.LogviewerAuthenticationConfiguration.LOGVIEWER_AUTHENTICATION_FILTER;


@Configuration
@EnableWebMvc
@Import({
        LogViewerConfiguration.class,
        LogviewerAuthenticationConfiguration.class,
        AuthorizationConfiguration.class,
        TvmConfiguration.class,
})
public class WebAppConfiguration implements WebMvcConfigurer {
    private final TvmInterceptor tvmInterceptor;

    @Autowired
    public WebAppConfiguration(TvmInterceptor tvmInterceptor) {
        this.tvmInterceptor = tvmInterceptor;
    }

    @Bean(name = "logviewerFilter")
    @Autowired
    public Filter logviewerFilter(
            TraceContextFilter traceContextFilter,
            AuthExceptionFilter authExceptionFilter,
            LogviewerLoggingFilter loggingFilter,
            CharacterEncodingFilter characterEncodingFilter,
            @Qualifier(LOGVIEWER_AUTHENTICATION_FILTER) FilterChainProxy filterChainProxy) {
        CompositeFilter compositeFilter = new CompositeFilter();
        compositeFilter.setFilters(Arrays.asList(
                characterEncodingFilter,
                traceContextFilter,
                authExceptionFilter,
                filterChainProxy,
                loggingFilter
        ));
        return compositeFilter;
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(tvmInterceptor);
    }

    @Bean
    public CharacterEncodingFilter characterEncodingFilter() {
        CharacterEncodingFilter characterEncodingFilter = new CharacterEncodingFilter();
        characterEncodingFilter.setForceEncoding(true);
        characterEncodingFilter.setEncoding("UTF-8");
        return characterEncodingFilter;
    }

    @Override
    public void configureMessageConverters(List<HttpMessageConverter<?>> converters) {
        Jackson2ObjectMapperBuilder builder = new Jackson2ObjectMapperBuilder();
        builder.indentOutput(true);
        builder.serializerByType(Timestamp.class, new JsonSerializer<Timestamp>() {
            private final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

            @Override
            public void serialize(Timestamp value, JsonGenerator gen, SerializerProvider serializers)
                    throws IOException {
                gen.writeString(value.toLocalDateTime().format(dateTimeFormatter));
            }
        });
        // javascript doesn't support 63-bits long, so we need to encode them as strings
        builder.serializerByType(Long.class, new JsonSerializer<Long>() {
            @Override
            public void serialize(Long value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
                gen.writeString(value.toString());
            }
        });
        converters.add(new MappingJackson2HttpMessageConverter(builder.build()));
        converters.add(new StringHttpMessageConverter());
    }

    @Override
    public void addResourceHandlers(ResourceHandlerRegistry registry) {
        registry.addResourceHandler("/logviewer/static/**")
                .addResourceLocations("/static/", "classpath:/logviewer/frontend_resource/")
                .setCacheControl(CacheControl.noStore());
    }
}
