package ru.yandex.direct.logviewer.controller;

import java.io.IOException;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletResponse;

import org.apache.poi.ss.usermodel.Workbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.jdbc.UncategorizedSQLException;
import org.springframework.security.access.annotation.Secured;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.clickhouse.except.ClickHouseErrorCode;
import ru.yandex.clickhouse.except.ClickHouseException;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.logviewer.auth.LogviewerUserRolesService;
import ru.yandex.direct.logviewer.domain.web.UpdateHistoryItemRequest;
import ru.yandex.direct.logviewer.utils.Prettifier;
import ru.yandex.direct.logviewercore.container.RowsWithCount;
import ru.yandex.direct.logviewercore.domain.Condition;
import ru.yandex.direct.logviewercore.domain.LogRecordInfo;
import ru.yandex.direct.logviewercore.domain.LogTablesInfoManager;
import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;
import ru.yandex.direct.logviewercore.domain.web.FilterResponse;
import ru.yandex.direct.logviewercore.domain.web.HistoryItem;
import ru.yandex.direct.logviewercore.domain.web.InfoResponse;
import ru.yandex.direct.logviewercore.domain.web.LogViewerFilterForm;
import ru.yandex.direct.logviewercore.service.CacheLogViewerResultService;
import ru.yandex.direct.logviewercore.service.LogViewerInvalidInputException;
import ru.yandex.direct.logviewercore.service.LogViewerService;
import ru.yandex.direct.logviewercore.service.LogViewerXlsService;
import ru.yandex.direct.web.core.JsonPropertyEditor;

import static org.springframework.http.HttpHeaders.CONTENT_DISPOSITION;
import static org.springframework.http.HttpHeaders.CONTENT_TYPE;
import static ru.yandex.direct.logviewer.auth.LogviewerUserRolesService.ROLE_LOGVIEWER_DIRECT;
import static ru.yandex.direct.logviewer.auth.LogviewerUserRolesService.ROLE_LOGVIEWER_GRUT;
import static ru.yandex.direct.logviewercore.service.LogViewerService.MAX_ROWS;

@Controller
@RequestMapping(path = {"/", "/logviewer"})
public class LogViewerController {
    private static final Logger logger = LoggerFactory.getLogger(LogViewerController.class);
    private static final String INDEX_PAGE = "/logviewer/static/index_original.html";
    private static final String NEW_INDEX_PAGE = "/logviewer/static/index.html";
    private static final String XLS_CONTENT_TYPE = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";

    @Target(ElementType.METHOD)
    @Retention(RetentionPolicy.RUNTIME)
    @Secured({"ROLE_SUPER", "ROLE_SUPERREADER", "ROLE_SUPPORT", "ROLE_LIMITED_SUPPORT", "ROLE_INTERNAL_AD_ADMIN",
            "ROLE_MANAGER", "ROLE_PLACER", ROLE_LOGVIEWER_DIRECT, ROLE_LOGVIEWER_GRUT})
    private @interface DefaultSecured {
    }

    @Autowired
    private LogviewerUserRolesService logviewerUserRolesService;

    @Autowired
    private LogViewerService logViewerService;

    @Autowired
    private LogViewerXlsService logViewerXlsService;

    @Autowired
    private CacheLogViewerResultService cacheService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        for (Class<?> clazz : List.of(LogViewerFilterForm.class)) {
            binder.registerCustomEditor(clazz, new JsonPropertyEditor<>(clazz));
        }
    }

    @RequestMapping({"/", ""})
    public String index() {
        User operator = getOperator();
        return logViewerService.isNewUiEnabled(operator) ? NEW_INDEX_PAGE : INDEX_PAGE;
    }

    @RequestMapping("/switchUi")
    @ResponseBody
    @DefaultSecured
    public void switchUi(@RequestParam("new") Boolean enableNew, HttpServletResponse resp) throws IOException {
        var operator = getOperator();
        boolean isSwitched = logViewerService.switchUi(operator, enableNew);
        if (isSwitched) {
            resp.setStatus(HttpServletResponse.SC_MOVED_TEMPORARILY);
            resp.setHeader("Location", "/logviewer");
        } else {
            resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            resp.getOutputStream().write("ERROR".getBytes(StandardCharsets.UTF_8));
        }
    }

    @RequestMapping("/short/{link}")
    public void shortLink(@PathVariable("link") String link, HttpServletResponse httpServletResponse) {
        httpServletResponse.setHeader("Location", "https://nda.ya.ru/t/" + link);
        httpServletResponse.setStatus(HttpServletResponse.SC_MOVED_TEMPORARILY);
    }

    @RequestMapping("/short/v2/{queryId}")
    public void shortLink(@PathVariable long queryId, HttpServletResponse httpServletResponse) {
        String queryForm = cacheService.getWebQueryForm(queryId);
        if (queryForm == null) {
            httpServletResponse.setStatus(HttpServletResponse.SC_NOT_FOUND);
            return;
        }
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss");
        StringBuilder location = new StringBuilder("/logviewer?queryId=")
                .append(queryId)
                .append("#")
                .append(queryForm)
                .append("$");
        httpServletResponse.setHeader("Location", location.toString());
        httpServletResponse.setStatus(HttpServletResponse.SC_MOVED_TEMPORARILY);
    }

    @RequestMapping("/info")
    @ResponseBody
    @DefaultSecured
    public InfoResponse info() {
        Set<String> userScopes = logviewerUserRolesService.getCurrentUserScopes();
        return logViewerService.getLogTypesInfo(logTable -> {
            String[] allowAccessFor = logTable.allowAccessFor();
            return logviewerUserRolesService.hasAccessToLog(userScopes, allowAccessFor);
        });
    }

    @RequestMapping("/filterXls/{logName}")
    @DefaultSecured
    public void filterXls(
            @PathVariable String logName,
            @RequestParam LogViewerFilterForm form,
            HttpServletResponse resp
    ) throws IOException {
        FilterResponse ret = internalRequestProcess(logName, form);

        if (ret.getError() != null) {
            resp.setStatus(HttpServletResponse.SC_BAD_REQUEST);
            resp.setHeader(CONTENT_TYPE, MediaType.TEXT_PLAIN_VALUE);
            resp.getOutputStream().write(
                    ret.getError().getBytes(StandardCharsets.UTF_8)
            );
        } else {
            resp.setHeader(CONTENT_TYPE, XLS_CONTENT_TYPE);
            String datetime = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss"));
            String filename = "logviewer-" + logName + "." + datetime + ".xlsx";
            resp.setHeader(CONTENT_DISPOSITION, "attachment; filename=\"" + filename + "\"");
            Workbook workbook = logViewerXlsService.createXlsReport(logName, form.getFields(), ret.getData());
            workbook.write(resp.getOutputStream());
        }
    }

    @RequestMapping("/queries")
    @ResponseBody
    @DefaultSecured
    public List<HistoryItem> getQueryHistory(
            @RequestParam(required = false, defaultValue = "0") long startAfter,
            @RequestParam(required = false, defaultValue = "0") long startBefore,
            @RequestParam(name = "status", required = false, defaultValue = "") List<String> statuses,
            @RequestParam(required = false, defaultValue = "") String logName,
            @RequestParam(required = false, defaultValue = "false") boolean onlyFavourite,
            @RequestParam(required = false, defaultValue = "0") int limit
    ) {
        startBefore = startBefore <= 0 ? System.currentTimeMillis() : startBefore;
        limit = limit <= 0 ? 1000 : limit;
        return cacheService.get(startAfter, startBefore,
                statuses, logName, onlyFavourite, limit);
    }

    @RequestMapping("/queries/cleanCache")
    @ResponseBody
    @Secured({"ROLE_SUPER", "ROLE_SUPERREADER"})
    public void cleanCache() {
        cacheService.clean();
    }

    @RequestMapping(path = "/query/{queryId}", method = RequestMethod.POST)
    @ResponseBody
    @DefaultSecured
    public void updateQueryInfo(
            @PathVariable long queryId,
            @RequestBody UpdateHistoryItemRequest request
    ) {
        cacheService.updateQuery(queryId, request.getName(), request.getShared(), request.getFavourite());
    }

    @RequestMapping("/query/{queryId}")
    @ResponseBody
    @DefaultSecured
    public HistoryItem getQueryResultById(
            @PathVariable long queryId,
            @RequestParam(required = false, defaultValue = "0") int offset,
            @RequestParam(required = false) @Nullable Integer limit,
            @RequestParam(required = false, defaultValue = "false") boolean recalculate
    ) {
        if (limit == null || limit <= 0 || offset < 0) {
            return cacheService.get(queryId);
        }
        HistoryItem item = cacheService.get(queryId, offset, limit);
        LogViewerFilterForm form = item.getRequest();
        if (form.getOffset() == offset && form.getLimit() == limit && !recalculate) {
            return item;
        }
        String logName = item.getLogName();
        form.setOffset(offset);
        form.setLimit(limit);

        long startTime = System.currentTimeMillis();
        FilterResponse response = null;
        cacheService.save(queryId, logName, form, response, startTime, 0);
        // Обновляем время старта, чтобы в executionTime не учитывалось время взаимодействия с ydb
        startTime = System.currentTimeMillis();
        try {
            response = internalRequestProcess(logName, form);
        } finally {
            long executionTime = System.currentTimeMillis() - startTime;
            cacheService.save(queryId, logName, form, response, startTime, (int) executionTime);
        }
        return cacheService.get(queryId, offset, limit);
    }

    @RequestMapping("/filter/{logName}")
    @ResponseBody
    @DefaultSecured
    public FilterResponse filter(
            @PathVariable String logName,
            @RequestParam LogViewerFilterForm form,
            @RequestParam(required = false, defaultValue = "false") boolean prettify
    ) {
        long startTime = System.currentTimeMillis();
        long queryId = cacheService.initHistoryRow(logName, form, startTime);
        FilterResponse response = null;
        try {
            response = internalRequestProcess(logName, form);
            response.setQueryId(queryId);
            if (prettify) {
                response = Prettifier.prettifyResponse(response, form);
            }
        } finally {
            long executionTime = System.currentTimeMillis() - startTime;
            cacheService.save(queryId, logName, form, response, startTime, (int) executionTime);
        }
        return response;
    }

    private FilterResponse internalRequestProcess(
            String logName,
            LogViewerFilterForm form
    ) {
        LogRecordInfo<? extends LogRecord> info = LogTablesInfoManager.getLogRecordInfo(logName);
        Set<String> scopes = logviewerUserRolesService.getCurrentUserScopes();
        if (!logviewerUserRolesService.hasAccessToLog(scopes, info.getAllowAccessFor())) {
            String login = logviewerUserRolesService.getCurrentUserLogin();
            logger.error("User {} with scopes {} has no access to log {}", login, scopes, logName);
            throw new AccessDeniedException("No access to log");
        }

        SimpleDb db;
        if (logName.startsWith("messages_grut")) {
            db = SimpleDb.CLICKHOUSE_GRUT_LOGS;
        } else {
            db = SimpleDb.CLICKHOUSE_CLOUD;
        }

        if (form.getFields().isEmpty()) {
            return new FilterResponse("No field is selected");
        }

        if (form.getOffset() < 0) {
            return new FilterResponse("Offset can't be negative");
        }

        try {
            Map<String, List<String>> enhancedConditions = logViewerService.enhanceConditions(form.getConditions());
            List<List<Object>> results;
            long totalCount = 0;
            String query;
            if (form.isShowStats()) {
                var statsRows = logViewerService.getLogStats(
                        db,
                        info,
                        logViewerService.devirtualizeFields(info, form.getFields()),
                        form.getFrom(),
                        form.getTo(),
                        enhancedConditions,
                        form.getLogTimeGroupBy(),
                        form.isSortByCount(),
                        form.isReverseOrder(),
                        form.getLimit(),
                        form.getOffset());
                results = statsRows.rows();
                query = statsRows.sql();
                logViewerService.enhanceResults(results, form.getFields());
            } else {
                List<String> formFields = logViewerService.devirtualizeFields(info, form.getFields());
                if (form.isShowTraceIdRelated()) {
                    if (!formFields.contains(info.getTraceIdColumn())) {
                        formFields.add(info.getTraceIdColumn());
                    }
                    if (!formFields.contains(info.getReqidColumn())) {
                        formFields.add(info.getReqidColumn());
                    }
                }
                RowsWithCount<? extends LogRecord> ret = logViewerService.getLogRows(
                        db,
                        info,
                        formFields,
                        form.getFrom(),
                        form.getTo(),
                        enhancedConditions,
                        form.getLimit(), form.getOffset(),
                        form.isReverseOrder());
                totalCount = ret.totalCount();
                if (form.isShowTraceIdRelated()) {
                    Map<String, List<Condition>> conditions = logViewerService.getTraceIdCondition(info, ret);
                    RowsWithCount<? extends LogRecord> traceIdSearchResult = logViewerService.getLogRowsWithConditions(
                            db,
                            info,
                            formFields,
                            form.getFrom(),
                            form.getTo(),
                            conditions,
                            MAX_ROWS, 0,
                            form.isReverseOrder());
                    //заполняем результат
                    List<LogRecord> rows =
                            logViewerService.mergeDataWithTraceIdRows(info, ret.rows(), traceIdSearchResult.rows());
                    ret = new RowsWithCount<>(totalCount, ret.query(), rows);
                }

                logViewerService.calculateVirtualColumns(info, form.getFields(), ret.rows());
                results = logViewerService.cutColumns(info, ret.rows(), form.getFields());
                logViewerService.enhanceResults(results, form.getFields());
                query = ret.query();
            }
            return new FilterResponse(results, totalCount, query);
        } catch (UncategorizedSQLException e) {
            String errorText = e.getMessage();
            if (ClickHouseException.class.isAssignableFrom(e.getSQLException().getClass())) {
                var errorCode = ClickHouseErrorCode.fromCode(e.getSQLException().getErrorCode());
                if (errorCode == ClickHouseErrorCode.TIMEOUT_EXCEEDED) {
                    errorText = "Query takes too long time";
                } else if (errorCode == ClickHouseErrorCode.MEMORY_LIMIT_EXCEEDED) {
                    errorText = "Query takes too much memory";
                } else {
                    errorText = e.getSQLException().getMessage();
                }
            }
            logger.error("Database error", e);
            return new FilterResponse(errorText);
        } catch (LogViewerInvalidInputException e) {
            logger.error("Invalid input", e);
            return new FilterResponse(e.getMessage());
        }
    }

    @Nullable
    private User getOperator() {
        var authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication instanceof DirectAuthentication) {
            DirectAuthentication directAuthentication = (DirectAuthentication) authentication;
            return directAuthentication.getOperator();
        }
        return null;
    }
}
