package ru.yandex.direct.logviewer.controller

import com.fasterxml.jackson.annotation.JsonProperty
import com.fasterxml.jackson.core.type.TypeReference
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Controller
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestMethod
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import ru.yandex.direct.logviewer.auth.LogviewerUserRolesService.Companion.IDM_ROLE_DIRECT
import ru.yandex.direct.logviewer.auth.LogviewerUserRolesService.Companion.IDM_ROLE_GRUT
import ru.yandex.direct.logviewer.auth.LogviewerUserRolesService.Companion.IDM_ROLE_SLUG
import ru.yandex.direct.logviewer.auth.LogviewerUserRolesTable
import ru.yandex.direct.logviewer.configuration.LogviewerYdbConfiguration.LOGVIEWER_YDB_CLIENT_BEAN
import ru.yandex.direct.logviewer.configuration.LogviewerYdbConfiguration.LOGVIEWER_YDB_PATH_BEAN
import ru.yandex.direct.tvm.AllowServices
import ru.yandex.direct.tvm.TvmService
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.upsertInto
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable
import javax.ws.rs.core.MediaType

data class LoginWithRole(
    val login: String,
    val role: String
)

class IdmInfoResponse(
    @JsonProperty("roles") val roles: IdmInfoResponseRoles
) {
    class IdmInfoResponseRoles(
        @JsonProperty("slug") val slug: String,
        @JsonProperty("name") val name: String,
        @JsonProperty("values") val values: Map<String, String>
    )

    @JsonProperty("code")
    val code = 0
}

class IdmGetAllRolesResponse(
    @JsonProperty("users") val users: List<IdmUserInfo>
) {
    class IdmUserInfo(
        @JsonProperty("login") val login: String,
        @JsonProperty("roles") val roles: List<Map<String, String>>
    )

    @JsonProperty("code")
    val code = 0
}

class IdmOkResponse {
    @JsonProperty("code")
    val code = 0
}

/**
 * Интеграция логвьювера с IDM
 */
@Controller
@RequestMapping(path = ["/idm"])
@AllowServices(
    production = [TvmService.IDM_PROD],
    testing = [TvmService.IDM_TEST, TvmService.DIRECT_DEVELOPER, TvmService.DIRECT_AUTOTESTS]
)
class LogViewerIdmController(
    @Qualifier(LOGVIEWER_YDB_CLIENT_BEAN) val ydbClient: YdbClient,
    @Qualifier(LOGVIEWER_YDB_PATH_BEAN) val path: YdbPath
) {
    companion object {
        private val ROLE_TYPE_REFERENCE: TypeReference<Map<String, String>> =
            object : TypeReference<Map<String, String>>() {}

        private val USER_ROLES = LogviewerUserRolesTable()
    }

    @RequestMapping(path = ["/info/"], method = [RequestMethod.GET], produces = [MediaType.APPLICATION_JSON])
    @ResponseBody
    fun info() = IdmInfoResponse(
        roles = IdmInfoResponse.IdmInfoResponseRoles(
            slug = IDM_ROLE_SLUG,
            name = "Просмотр логов",
            values = mapOf(
                IDM_ROLE_DIRECT to "Доступ к логам Директа",
                IDM_ROLE_GRUT to "Доступ к логам GrUT"
            )
        )
    )

    @ResponseBody
    @RequestMapping(
        path = ["/get-all-roles/"],
        method = [RequestMethod.GET],
        produces = [MediaType.APPLICATION_JSON]
    )
    fun getRoles(): IdmGetAllRolesResponse {
        val queryAndParams = SelectBuilder.select(
            USER_ROLES.LOGIN,
            USER_ROLES.ROLE
        )
            .from(USER_ROLES)
            .queryAndParams(path)
        val reader = ydbClient.executeQuery(queryAndParams).getResultSet(0)

        val userRoles = mutableListOf<LoginWithRole>()
        while (reader.next()) {
            val login = reader.getValueReader(USER_ROLES.LOGIN).utf8
            val role = reader.getValueReader(USER_ROLES.ROLE).utf8
            userRoles.add(LoginWithRole(login, role))
        }

        val users = userRoles
            .groupBy({ it.login }, { it.role })
            .map { (login, roles) ->
                IdmGetAllRolesResponse.IdmUserInfo(login, roles.map { mapOf(IDM_ROLE_SLUG to it) })
            }

        return IdmGetAllRolesResponse(users)
    }

    @RequestMapping(
        path = ["/add-role/"],
        method = [RequestMethod.POST],
        consumes = [MediaType.APPLICATION_FORM_URLENCODED],
        produces = [MediaType.APPLICATION_JSON]
    )
    @ResponseBody
    fun addRole(
        @RequestParam("login") login: String,
        @RequestParam("role") jsonRole: String
    ): IdmOkResponse {
        val roleName = JsonUtils.fromJson(jsonRole, ROLE_TYPE_REFERENCE)[IDM_ROLE_SLUG]
        ydbClient.executeQuery(
            upsertInto(USER_ROLES)
                .selectAll()
                .from(buildTempTable {
                    value(USER_ROLES.LOGIN, login)
                    value(USER_ROLES.ROLE, roleName)
                })
                .queryAndParams(path)
        )
        return IdmOkResponse()
    }

    @RequestMapping(
        path = ["/remove-role/"],
        method = [RequestMethod.POST],
        consumes = [MediaType.APPLICATION_FORM_URLENCODED],
        produces = [MediaType.APPLICATION_JSON]
    )
    @ResponseBody
    fun removeRole(
        @RequestParam("login") login: String,
        @RequestParam("role") jsonRole: String
    ): IdmOkResponse {
        val roleName = JsonUtils.fromJson(jsonRole, ROLE_TYPE_REFERENCE)[IDM_ROLE_SLUG]
        ydbClient.executeQuery(
            deleteFrom(USER_ROLES)
                .where(USER_ROLES.LOGIN.eq(login).and(USER_ROLES.ROLE.eq(roleName)))
                .queryAndParams(path)
        )
        return IdmOkResponse()
    }
}
