package ru.yandex.direct.logviewer.utils;

import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.logviewercore.domain.web.FilterResponse;
import ru.yandex.direct.logviewercore.domain.web.LogViewerFilterForm;

public class Prettifier {

    private static final DecimalFormat DECIMAL_FORMAT = new DecimalFormat("0.000");
    private static final Set<String> COLUMNS_FOR_DECIMAL_FORMAT = Set.of(
            "runtime", "span_time", "ela", "cpu_system", "cpu_user", "mem");

    private static final String COUNT_COLUMN = "_count";
    private static final NumberFormat COUNT_FORMAT = setGroupingSeparatorToFormat(new DecimalFormat("#"), ' ');
    private static final SimpleDateFormat SIMPLE_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    private static final Set<String> COLUMNS_FOR_JSON_FORMAT = Set.of(
            "param", "message", "data", "row");

    private static final Gson JSON_PRETTY = new GsonBuilder().setPrettyPrinting().create();
    private static final Gson JSON_FOR_URL = new GsonBuilder().create();

    private static final String JSON_FIELD_JSON = "_json";
    private static final String JSON_FIELD_LOGIN = "ulogin";
    private static final String JSON_FIELD_OPERATION = "operationName";
    private static final String JSON_FIELD_QUERY = "query";
    private static final String JSON_FIELD_VARIABLES = "variables";

    private static final Pattern STACK_TRACE_LINE_PATTERN = Pattern.compile("\\|(\t*(at |Suppressed: |Caused by: |\\.\\.\\. \\d+ more))");
    private static final Pattern GRID_QUERY_SEPARATORS_PATTERN = Pattern.compile("[() {}]");
    private static final Pattern MIDDLE_EMPTY_LINES = Pattern.compile("\\n(\\s+\\n)+");
    private static final Pattern EDGE_EMPTY_LINES = Pattern.compile("^(\\s*\\n)+|(\\n\\s*)+$");

    private static final String FULL_VALUE = "full";
    private static final String COLLAPSE_VALUE = "collapse";
    private static final String WEB_API_GRID = "webApiGrid";

    private static final Logger logger = LoggerFactory.getLogger(Prettifier.class);

    private static DecimalFormat setGroupingSeparatorToFormat(DecimalFormat format, char groupingSeparator) {
        DecimalFormatSymbols dfs = format.getDecimalFormatSymbols();
        dfs.setGroupingSeparator(groupingSeparator);
        format.setDecimalFormatSymbols(dfs);
        return format;
    }

    public static FilterResponse prettifyResponse(FilterResponse response, LogViewerFilterForm form) {
        if (response.getData() != null) {
            List<String> columns = new ArrayList<>(form.getFields());
            StreamEx.of(response.getData())
                    .nonNull()
                    .filter(Predicate.not(List::isEmpty))
                    .forEach(row -> prettifyResultRow(row, columns));
        }
        return response;
    }

    private static void prettifyResultRow(List<Object> data, List<String> columns) {
        if (data.size() > columns.size()) {
            columns.add(COUNT_COLUMN);
        }
        List<Object> newValues = EntryStream.zip(columns, data)
                .mapToValue((column, value) -> dataToString(column, value))
                .mapToValue((column, value) -> prettifyStringValue(column, value))
                .values()
                .toList();
        data.clear();
        data.addAll(newValues);
    }

    static String dataToString(String column, Object data) {
        if (data == null) {
            return "";
        } else if (data.getClass().isArray()) {
            if (data.getClass().equals(int[].class)) {
                data = Arrays.stream((int[]) data).boxed().toArray();
            } else if (data.getClass().equals(long[].class)) {
                data = Arrays.stream((long[]) data).boxed().toArray();
            } else if (data.getClass().equals(double[].class)) {
                data = Arrays.stream((double[]) data).boxed().toArray();
            } else if (data.getClass().getComponentType().isPrimitive()) {
                return data.toString(); // Сейчас нет массивов из других примитавных типов
            }
            return StreamEx.of((Object[])data).map(Object::toString).joining(", ");
        } else if (COUNT_COLUMN.equals(column)) {
            return COUNT_FORMAT.format(data);
        } else if ((data.getClass().equals(Double.class) || data.getClass().equals(Float.class)) &&
                COLUMNS_FOR_DECIMAL_FORMAT.contains(column)) {
            return DECIMAL_FORMAT.format(data);
        } else if (data.getClass().equals(Timestamp.class)) {
            return SIMPLE_DATE_FORMAT.format(data);
        }
        return data.toString();
    }

    private static Object prettifyStringValue(String column, String data) {
        if (COLUMNS_FOR_JSON_FORMAT.contains(column) &&
                (data.startsWith("{") || data.startsWith("["))
        ) {
            JsonElement json = parseAndTransformJson(data);
            if (json != null) {
                Map<String, String> gridQuery = parseGridQuery(json);
                if (gridQuery != null) {
                    return gridQuery;
                }
                data = JSON_PRETTY.toJson(json);
                data = EDGE_EMPTY_LINES.matcher(data).replaceAll("");
            }
        } else if ("message".equals(column)) {
            data = prettifyStackTrace(data);
        }

        String[] lines = data.split("\n", 3);
        if (lines.length > 2) {
            return Map.of(
                    FULL_VALUE, data,
                    COLLAPSE_VALUE, lines[0] + "\n" + lines[1] + "..."
            );
        }
        if (data.length() > 1000) {
            return Map.of(
                    FULL_VALUE, data,
                    COLLAPSE_VALUE, data.substring(0, 997) + "..."
            );
        }
        return data;
    }

    /**
     * Парсим JSON.
     * По возможности трансформируем поле '_json' в JSON.
     * @return null если невалидный json
     */
    private static JsonElement parseAndTransformJson(String jsonString) {
        JsonElement json = null;
        try {
            json =  new JsonParser().parse(jsonString);
            JsonObject jsonObject = json.getAsJsonObject();
            String subJson = jsonObject.getAsJsonPrimitive(JSON_FIELD_JSON).getAsString();
            jsonObject.add(JSON_FIELD_JSON, new JsonParser().parse(subJson));
        } catch (RuntimeException ex) {
            // Если json невалидный, то вернум null
            // Если _json и так является json-ом (а не строкой), то трансформация не нужна.
        }
        return json;
    }

    /**
     * Выводим гридовый запрос в удобном формате
     * Добавляем сокращенную форму
     * Добавляем ссылку на запуск запроса в /web-api/grid
     */
    private static Map<String, String> parseGridQuery(JsonElement json) {
        try {
            String login = json.getAsJsonObject().getAsJsonPrimitive(JSON_FIELD_LOGIN).getAsString();
            JsonObject subJson = json.getAsJsonObject().getAsJsonObject(JSON_FIELD_JSON);
            String operation = subJson.getAsJsonPrimitive(JSON_FIELD_OPERATION).getAsString();
            JsonElement variables = subJson.get(JSON_FIELD_VARIABLES);
            String query = subJson.getAsJsonPrimitive(JSON_FIELD_QUERY).getAsString();

            Map<String, String> result = new HashMap<>(3);
            result.put(FULL_VALUE, formatGridQuery(query) + "\n\nvariables: " + JSON_PRETTY.toJson(variables));
            result.put(COLLAPSE_VALUE, operation + " {...}");
            try {
                URI uri = new URIBuilder()
                        .setPath("/web-api/grid/")
                        .addParameter(JSON_FIELD_OPERATION, operation)
                        .addParameter("user_login", login)
                        .addParameter(JSON_FIELD_VARIABLES, JSON_FOR_URL.toJson(variables))
                        .setFragment(query)
                        .build();
                result.put(WEB_API_GRID, uri.toString());
            } catch (URISyntaxException ex) {
                logger.warn("Can't build link to web api grid for logviewer", ex);
            }
            return result;
        } catch (RuntimeException ex) {
            logger.debug(String.format("Not Grid Query: %s", json), ex);
            return null;
        }
    }

    private static String formatGridQuery(String query) {
        final Set<String> prefixes = Set.of("...", "on", "...on");
        if (query.contains("\n")) {
            return query;
        }
        Matcher matcher = GRID_QUERY_SEPARATORS_PATTERN.matcher(query);
        if (!matcher.find()) {
            logger.warn("Can't prettify grid query " + query);
            return query;
        }
        int index = matcher.start();
        String shift = "";
        boolean isBracketOpened = false;
        StringBuilder result = new StringBuilder(query.substring(0, index + 1));
        while (matcher.find()) {
            String lastWord = query.substring(index + 1, matcher.start());
            result.append(lastWord);
            index = matcher.start();
            char symbol = query.charAt(index);
            if (isBracketOpened) {
                result.append(symbol);
                isBracketOpened = symbol != ')';
            } else {
                switch (symbol) {
                    case ' ': {
                        if (prefixes.contains(lastWord)) {
                            result.append(symbol);
                        } else {
                            result.append("\n").append(shift);
                        }
                        continue;
                    }
                    case '{': {
                        shift = shift + "  ";
                        result.append(symbol).append("\n").append(shift);
                        continue;
                    }
                    case '}': {
                        shift = shift.length() > 2 ? shift.substring(2) : "";
                        result.append("\n").append(shift).append(symbol).append("\n").append(shift);
                        continue;
                    }
                    case '(': {
                        isBracketOpened = true;
                    }
                    default:
                        result.append(symbol);
                }
            }
        }
        result.append(query.substring(index + 1));
        String formatQuery = MIDDLE_EMPTY_LINES.matcher(result.toString()).replaceAll("\n");
        formatQuery = EDGE_EMPTY_LINES.matcher(formatQuery).replaceAll("");
        return formatQuery;
    }

    private static String prettifyStackTrace(String stackTraceString) {
        return STACK_TRACE_LINE_PATTERN.matcher(stackTraceString).replaceAll("\n$1");
    }
}
