var logViewerApp = (function () {
    var self;
    var urlBase = "/logviewer";
    var config;

    var $logTypeSelect;
    var $logFieldCheckboxes, $logFieldConditions, $logShowBtn, $columnList, $result, $btnShortLink;
    var $showStatsCheckbox, $logTimeGroupBySelect, $sortByCountCheckbox, $traceidRequests, $traceidRelatedCheckbox;
    var $pageSize, $toggleParams, $showParamsUncollapsed, $showSql, $reverseOrder;
    var $xlsLink;
    var $currentRequest;
    var $shortLinkModal, $shortLinkModalInput, $shortLinkModalCopyButton;
    return {
        init: function () {
            self = this;

            $logTypeSelect = $("select#logType");
            $logTypeSelect.on("change", self.onLogTypeChanged);

            $logFieldCheckboxes = $("#logFieldCheckboxes");
            $logFieldCheckboxes.on("change", "input[type=checkbox]", self.onLogFieldChecked);

            $logFieldConditions = $("#logFieldConditions");
            $pageSize = $("#pageSize");
            $logShowBtn = $("#logForm button[type=submit]");
            $btnShortLink = $("#btn_short_link");
            $shortLinkModal = $("#short_link_modal");
            $shortLinkModalInput = $("#short_link_modal_input");
            $shortLinkModalCopyButton = $("#short_link_modal_copy_button");

            $("#logForm").on("submit", function (e) {
                e.preventDefault();
                self.showLogs(1, 1 * $pageSize.val(), "html");
            });

            $showStatsCheckbox = $("#show_stats");

            $showParamsUncollapsed = $('#show_params_uncollapsed');
            $showParamsUncollapsed.on("change", null, function(e) {
                if ($showParamsUncollapsed.prop('checked')) {
                    $('.short-param.has-full').addClass('hidden');
                    $('.full-param').removeClass('hidden');
                } else {
                    $('.short-param.has-full').removeClass('hidden');
                    $('.full-param').addClass('hidden');
                }
            });

            $showSql = $('#show_sql');
            $showSql.on("change", null, function (e) {
                $('#sql_block').toggleClass('hidden', !$showSql.prop('checked'));
            });

            $xlsLink = $('#xls_link');
            $xlsLink.on("click", function (e) {
                try {
                    self.showLogs(1, 1 * $pageSize.val(), "xls");
                } catch (e) {
                    console.log(e);
                    alert("internal error");
                    e.preventDefault();
                }
            });

            $logTimeGroupBySelect = $("#log_time_group_by");
            $sortByCountCheckbox = $("#sort_by_count");
            $traceidRequests = $("#traceid_requests");
            $traceidRelatedCheckbox = $("#show_traceid_related");
            $reverseOrder = $('#reverse_order');

            $columnList = $("#columnList");
            $columnList.on("click", "ul > li > a", self.onColumnListChanged);
            $columnList.on("click", "ul button", self.onColumnListButtonClick);

            $result = $("#result");
            $result.on("click", "a[data-page]", function () {
                self.showLogs($(this).data("page"), 1 * $(this).data("page-size"), "html");
                return false;
            });

            $(window).on("hashchange", function () {
                self.loadHistory();
            });

            self.initDateRangePicker();
            self.loadConfig(function () {
                self.loadHistory();
            });

            $('[data-toggle="tooltip"]').tooltip();

            $btnShortLink.on("click", function () {
                self.makeShortLink()
            });
            $shortLinkModalCopyButton.on("click", function () {
                self.shortLinkModalCopyButtonClick()
            });
        },

        applyDateRange: function (start, end) {
            $("input#logRangeFrom").val(start.format('YYYY-MM-DDTHH:mm:ss'));
            $("input#logRangeTo").val(end.format('YYYY-MM-DDTHH:mm:ss'));
        },

        initDateRangePicker: function () {
            $("#logRange").daterangepicker({
                "timePicker": true,
                "timePicker24Hour": true,
                "timePickerSeconds": true,
                "alwaysShowCalendars": true,
                locale: {
                    format: 'ddd D MMM YYYY HH:mm:ss',
                    firstDay: 1,
                },
                ranges: {
                    'Last Hour': [moment().subtract(1, 'hours'), moment().endOf('day')],
                    'Today': [moment().startOf('day'), moment().endOf('day')],
                    'Yesterday': [moment().subtract(1, 'days').startOf('day'), moment().subtract(1, 'days').endOf('day')],
                    'Last 3 Days': [moment().subtract(2, 'days').startOf('day'), moment().endOf('day')],
                    'Last Week': [moment().subtract(6, 'days').startOf('day'), moment().endOf('day')],
                    'Last Month': [moment().subtract(1, 'months').startOf('day'), moment().endOf('day')],
                    'Last 6 Months': [moment().subtract(6, 'months').startOf('day'), moment().endOf('day')],
                    'Last Year': [moment().subtract(1, 'years').startOf('day'), moment().endOf('day')],
                },
            }, self.applyDateRange);
            self.applyDateRange(moment().startOf('day'), moment().endOf('day'));
        },

        loadConfig: function (afterCb) {
            $.get(urlBase + "/info", "", function (data) {
                config = {logs: {}};

                $logTypeSelect.empty();
                $logTypeSelect.append("<option selected>Choose...</option>");
                data.logs.forEach(function (item) {
                    config.logs[item.name] = item;
                    $logTypeSelect.append("<option value=\"" + item.name + "\">" +
                        item.name +
                        (item.desc ? " - " + item.desc : "") +
                        "</option>");
                });

                if (afterCb) {
                    afterCb();
                }
            }).fail(function (data) {
                alert("Error: " + data.responseText);
            });
        },

        getLogConfig: function (logType) {
            if (!logType) {
                return null;
            }
            return config.logs[logType];
        },

        currentLogConfig: function () {
            var logType = $logTypeSelect.val();
            return self.getLogConfig(logType)
        },

        getFieldConditions: function () {
            var conditions = {};

            var logType = $logTypeSelect.val();
            var logConfig = self.getLogConfig(logType);
            var colScales = {};
            logConfig.columns.forEach(function (col) {
                if (col.scale) {
                    colScales[col.name] = col.scale;
                }
            });

            $("input[data-field]", $logFieldConditions).each(function () {
                var field = $(this).data("field");
                var val = $(this).val();

                if (val != null && val.match("\\S")) {
                    if (colScales[field]) {
                        val = (val * colScales[field]).toString();
                    }
                    conditions[field] = val;
                }
            });
            return conditions;
        },

        setFieldConditions: function (data) {
            // Убрать все, для правильно работы перехода "назад" в браузере
            $("input[data-field]", $("#logFieldCheckboxes")).prop('checked', false).change();

            Object.keys(data).forEach(function (field) {
                var val = data[field];
                self.setFieldCheckbox(field, true);
                $("input[data-field=\"" + field + "\"]", $logFieldConditions).val(val);
            });
        },

        onLogTypeChanged: function () {
            var prevFieldConditions = self.getFieldConditions();

            $("ul li", $columnList).remove();
            $logFieldCheckboxes.empty();
            $logFieldConditions.empty();
            $result.empty();

            var logType = $logTypeSelect.val();
            var logConfig = self.getLogConfig(logType);
            if (!logConfig) {
                $("button", $columnList).addClass("disabled");
                $("#logForm .form-group").has($logFieldCheckboxes).addClass("hidden");
                $logShowBtn.addClass("disabled");
                location.hash = '';
                return;
            }

            $("button", $columnList).removeClass("disabled");
            $("#logForm .form-group").has($logFieldCheckboxes).removeClass("hidden");
            $logShowBtn.removeClass("disabled");
            // Для логов, у которых trace_id скрыто (strictlyHidden == true),
            // не нужно отображать чекбокс "show all traceIds"
            var traceIdCol = logConfig.columns.find(val =>
                val.name == logConfig.traceIdColumn && logConfig.traceIdColumn !== ""
            );
            if (traceIdCol != null && !traceIdCol.strictlyHidden) {
                $traceidRequests.removeClass("invisible");
            } else {
                $traceidRequests.addClass("invisible");
                $traceidRelatedCheckbox.prop("checked", false);
            }

            var fieldTemplate = $("#logFieldCheckboxTmpl").html();
            var colTemplate = $("#columnTmpl").html();
            var showCols = logConfig.columns.filter(val => !val.strictlyHidden);
            $("ul", $columnList).append(Mustache.render(colTemplate, {cols: showCols}));
            logConfig.columns.filter(function (val) {
                return !val.virtual;
            }).forEach(function (col) {
                $logFieldCheckboxes.append(Mustache.render(fieldTemplate, col));
            });

            self.setFieldConditions(prevFieldConditions);
        },

        onColumnListChanged: function () {
            $("span", $(this)).toggleClass("invisible");
            return false;
        },

        onColumnListButtonClick: function (event) {
            if (this.innerText === 'Default') {
                var colByName = {};
                self.currentLogConfig().columns.forEach(function (col) {
                    colByName[col.name] = col
                });

                $(this).parents('ul').find('li a').each(function (i, el) {
                    var field = $(this).data("field");
                    $(this).find('.glyphicon-ok')[colByName[field].hidden ? 'addClass' : 'removeClass']('invisible')
                });
            } else {
                var method = {
                    'All': 'removeClass',
                    'None': 'addClass',
                    'Invert': 'toggleClass'
                }[this.innerText];
                $(this).parents('ul').find('.glyphicon-ok')[method]('invisible');
            }
            return false;
        },

        onLogFieldChecked: function () {
            var $el = $(this);
            var field = $el.data("field");
            var checked = $el.prop("checked");

            if (checked) {
                self.addLogField(field).focus();
            } else {
                self.removeLogField(field);
            }
        },

        addLogField: function (field) {
            var $lf = $("#logField-" + field);
            if (!$lf.length) {
                var template = $("#logFieldTmpl").html();
                $logFieldConditions.append(Mustache.render(template, {name: field}));
                $lf = $("#logField-" + field);
                $lf.closest("div.form-group").find("button.close").on("click", function (e) {
                    self.setFieldCheckbox(field, false);
                });
            }
            return $lf;
        },

        removeLogField: function (field) {
            $(".form-group", $logFieldConditions).has("input[data-field=\"" + field + "\"]").remove();
        },

        setFieldCheckbox: function (field, flag) {
            $("input[data-field=\"" + field + "\"]", $logFieldCheckboxes).prop("checked", flag).change();
        },

        showLogs: function (page, pageSize, format) {
            if ($logShowBtn.hasClass("disabled")) {
                return false;
            }

            var logType = $logTypeSelect.val();
            var logConfig = self.getLogConfig(logType);
            if (!logConfig) {
                return false;
            }

            var fields = [];
            $("ul > li > a[data-field]", $columnList).each(function () {
                if (!$("span", $(this)).hasClass("invisible")) {
                    fields.push($(this).data("field"));
                }
            });

            page = Math.max(1, parseInt(page) || 1);
            var form = {
                from: $("input#logRangeFrom").val(),
                to: $("input#logRangeTo").val(),
                fields: fields,
                conditions: self.getFieldConditions(),
                limit: pageSize,
                offset: (page - 1) * pageSize,
                reverseOrder: $reverseOrder.prop("checked"),
                showTraceIdRelated: $traceidRelatedCheckbox.prop("checked"),
            };

            if ($showStatsCheckbox.prop("checked")) {
                if (form.fields.length > 3) {
                    alert("You can choose not more than 3 columns to group by stats");
                    return;
                }
                // if (logConfig.columns.some(function (c) {
                //     return c.heavy && form.fields.indexOf(c.name) >= 0;
                // })) {
                //     alert("We can't group logs by heavy column to show stats");
                //     return;
                // }
                form.showStats = true;
                form.sortByCount = $sortByCountCheckbox.prop("checked");
                if ($logTimeGroupBySelect.val()) {
                    if (!form.fields.some(function (x) {
                        return x == 'log_time' || x == 'datetime' || x == 'time_received' || x == 'time'
                    })) {
                        alert("You should choose log_time or datetime column to group by it");
                        return;
                    }
                    form.logTimeGroupBy = $logTimeGroupBySelect.val();
                }
            }

            self.saveHistory(logType, form);

            if ($currentRequest) {
                $currentRequest.abort();
            }

            if (format === "xls") {
                $xlsLink.attr({
                    "href": urlBase + "/filterXls/" + logType + "?form=" + encodeURIComponent(JSON.stringify(form)) + "&rnd=" + Math.random()
                });
                return true;
            }

            $result.html("Loading...");

            $currentRequest = $.get(urlBase + "/filter/" + logType, {form: JSON.stringify(form)})
                .done(function (res) {
                    if (res.error) {
                        $result.html("ERROR: " + res.error + ", reqid=" + res.reqid);
                    } else {
                        self.renderResult(form, res);
                    }
                })
                .fail(function (x) {
                    if (x.status == '402') {
                        $result.html("ERROR: incorrect authentication, please relogin (" + x.responseText + ")");
                        var data = $.parseJSON(x.responseText);
                        if (data.url_base) {
                            window.location.href = data.url_base + encodeURIComponent(window.location.href);
                        }
                    } else {
                        $result.html("ERROR: see server logs (" + x.status + " " + x.statusText + ")" + x.data);
                    }
                })
                .always(function () {
                    $currentRequest = null;
                });
            return false;
        },

        makeShortLink: function () {
            if ($logShowBtn.hasClass("disabled")) {
                console.log("no short link, $logShowBtn is disabled");
                self.showTooltipOnce($btnShortLink, "No link to copy");
                return;
            }

            var link = window.location.href;
            console.log('link: ', link);
            if (!link.endsWith('$')) {
                console.log("no short link, link not ends with $");
                self.showTooltipOnce($btnShortLink, "No link to copy");
                return;
            }
            console.log("started requesting short link");
            $.ajax({
                url:"https://nda.ya.ru/--?url=" + encodeURIComponent(link),
                timeout: 1000
            }).complete(function (response) {
                if (response.status !== 200) {
                    console.log("error from nda.ya.ru");
                    self.showTooltipOnce($btnShortLink, "Error");
                    return
                }
                var shortLink = response.responseText;
                if (!shortLink.includes("https://nda.ya.ru/t/")) {
                    console.log("Short link doesn't contain https://nda.ya.ru/t/", shortLink);
                    self.showTooltipOnce($btnShortLink, "Error");
                    return
                }
                shortLink = location.protocol + '//' + location.host + "/logviewer/short/"
                    + shortLink.replace("https://nda.ya.ru/t/", "");

                console.log("short link to copy: " + shortLink);
                var copied  = self.copyToClipboard(shortLink);
                if (copied) {
                    console.log("copied short link");
                    self.showTooltipOnce($btnShortLink, "Copied");
                } else {
                    // for safari and old browsers
                    $shortLinkModalInput.val(shortLink);
                    $shortLinkModal.modal({showClose: false});
                    $shortLinkModalInput.select();
                }
            });
        },

        // for safari and old browsers
        shortLinkModalCopyButtonClick: function() {
            $shortLinkModalInput.select();
            var copied = document.execCommand('copy');
            if (copied) {
                $.modal.close()
            }
        },

        showTooltipOnce: function (element, text) {
            if (typeof element.data('title_backup') == "undefined") {
                element.data('title_backup', element.attr('title'))
            }
            element.attr('title', text);
            element.tooltip('show');
            element.on('hidden.bs.tooltip', () =>
            {
                element.off('hidden.bs.tooltip');
                element.tooltip('destroy');
                element.attr('title', element.data('title_backup'));
                element.tooltip();
            });
        },

        objGetOrDefault: function (from, key, defValue) {
            if (from.hasOwnProperty(key)) {
                return from[key];
            }
            return defValue;
        },

        isGridRequest: function (obj) {
            return obj.hasOwnProperty('_json')
                && obj._json.hasOwnProperty("operationName")
                && obj._json.hasOwnProperty("query");
        },

        formatGrid: function (obj) {
            var _json = obj._json

            var opName = _json.operationName;

            // удаляем строки с __typename
            _json.query = _json.query.replace(/\n\s+__typename\b/g, '');

            // получаем параметры
            var args = self.objGetOrDefault(_json, "variables", {});

            // если пусто, ничего делать не надо
            if (Object.getOwnPropertyNames(args).length != 0) {

                // получаем строку определения
                var rawDefMatch = _json.query.match(new RegExp(opName + "\\([^\\)]*\\)"));
                if (!rawDefMatch) {
                    // что-то пошло не так
                    console.error("Couldn't match function definition " + _json.query);
                    return self.formatNotGrid(obj);
                }
                var rawDef = rawDefMatch[0];
                var paramsDelimiter = "\n\t\t";

                // подставляем значения параметров
                var params = rawDef
                    .substring(opName.length + 1, rawDef.length - 1) // только аргументы
                    .split(',')
                    .map(s => s.trim())
                    .map(function (str) {

                        // находим имя вида $NAME: и ищем NAME в аргументах
                        var colonIdx = str.indexOf(':');
                        var varName = str.substring(1, colonIdx).trim();

                        if (!args.hasOwnProperty(varName)) {
                            throw "Argument " + varName + " in " + opName + " not found in variables!";
                        }

                        return str + " = " + JSON.stringify(args[varName]);
                    })
                    .join(',' + paramsDelimiter);

                var newDefinition = opName + " (" + paramsDelimiter + params + "\n)";

                _json.query = _json.query.substring(0, rawDefMatch.index)
                        + newDefinition
                        + _json.query.substring(rawDefMatch.index + rawDef.length);
            }
            return _json.query;
        },

        formatNotGrid: function(obj) {
            var text = json_bigint.stringify(obj, null, 2);
            // убираем лишний отступ в 2 пробела в начале каждой строки
            text = text.replace(/^  /gm, "");
            // схлопываем "}, \n {" в одну строку
            text = text.replace(/\},\n\s+\{/g, '}, {');
            text = text.replace(/^([\{\[])\n/, '$1').replace(/\n([\}\]])$/, '$1');
            text = text.replace(/\\n/g, '\n');
            return text;
        },

        renderResult: function (form, res) {
            var template = $("#resultTmpl").html();

            var fields = form.fields;

            var logType = $logTypeSelect.val();

            var logConfig = self.getLogConfig(logType);
            var colInfo = {};
            logConfig.columns.forEach(function (col) {
                colInfo[col.name] = col;
            });

            var rows = res.data.map(function (row, rowidx) {
                return [{
                    field: '_index',
                    value: (1 + form.offset + rowidx) + ".",
                    is_index: true
                }].concat(row.map(function (col, idx) {
                    var isGridInParamField = false;
                    var isFilterable = true;

                    var field = fields[idx] || '_count',
                        ret = {},
                        longColArray = [];

                    isFilterable = isFilterable && field !== '_count' &&
                        col !== null && col !== "" &&
                        colInfo[field] && !colInfo[field].virtual;

                    if (colInfo[field] && colInfo[field]['scale']) {
                        col = (col / colInfo[field]['scale']).toString();
                    } else if (col === null) {
                        col = "";
                    } else if (
                        (field === 'runtime'
                            || field === 'span_time'
                            || field === 'ela'
                            || field === 'cpu_system'
                            || field === 'cpu_user'
                            || field === 'mem')
                        && typeof col === "number") {
                        // время выпонения запроса
                        col = col.toFixed(3);
                    } else if (field === '_count') {
                        // показ статистики
                        col = col.toString().replace(/(\d)(?=(\d{3})+(?!\d))/g, "$1_");
                    } else if ($.isArray(col)) {
                        // джойним массив с пробелом (по умолчанию будет без пробела)
                        // чтобы работал перенос строк
                        col = col.join(', ');
                    } else {
                        col = "" + col;
                        if (col.match(/^[\[\{]/) && (field === 'param' || field === 'message' || field === 'data' || field === 'row')) {
                            try {
                                var obj = json_bigint.parse(col);
                                if (obj.hasOwnProperty('_json')) {
                                    try {
                                        obj._json = json_bigint.parse(obj._json);
                                    } catch (e) {
                                        console.error(e);
                                    }
                                }
                                var text;

                                if(self.isGridRequest(obj)) {
                                    text = self.formatGrid(obj);
                                    isGridInParamField = true;
                                } else {
                                    text = self.formatNotGrid(obj);
                                }
                                col = text;
                            } catch (e) {
                                console.error(e);
                                col = col.replace(/("|null|\}|\]),([^\\])/g, '$1,\n$2');
                            }
                        } else {
                            if (field === 'message') {
                                col = col.replace(/\|(\t*(at |Suppressed: |Caused by: |\.\.\. \d+ more))/g, "\n$1");
                            }
                            col = col.replace(/("|null|\}|\]),([^\\])/g, '$1,\n$2');
                        }
                        col = col.replace(/(^\n+|\n+$)/g, "").replace(/\n\n+/g, "\n");
                    }
                    longColArray = col.split('\n');

                    isFilterable = isFilterable && !col.match(/\n/);
                    isReqid = field === logConfig.reqidColumn;
                    isTraceId = field === logConfig.traceIdColumn;
                    let reqidType;
                    if (isTraceId) {
                        reqidType = "traceId";
                    } else if (isReqid) {
                        // не используется, но для порядка заполним
                        reqidType = "reqid";
                    } else {
                        reqidType = undefined;
                    }

                    ret = {
                        field: field,
                        value: col,
                        useReqidNavigation: isReqid || isTraceId,
                        reqidType: reqidType,
                        is_filterable: isFilterable
                    };


                    if (longColArray.length > 2) {
                        ret.value = longColArray.slice(0, 2).join('\n');
                        ret.valueFull = col;
                    } else if (field === 'cid' && col.length > 80) {
                        ret.value = col.substr(0, 60);
                        ret.valueFull = col;
                    } else if (field === 'class_name') {
                        ret.value = self.shortClassName(col, 40);
                        if (ret.value !== col) {
                            ret.valueFull = col;
                        }
                    } else if (field === 'remote_addr') {
                        ret.value = col.substr(0, 15);
                        if (ret.value !== col) {
                            ret.valueFull = col;
                        }
                    } else if (field === 'req_uri') {
                        ret.value = col.substr(0, 60);
                        if (ret.value !== col) {
                            ret.valueFull = col;
                        }
                    } else if (field === 'aggr_host_name' || field === 'aggr_service_name'
                        || field === 'child_host_name' || field === 'child_service_name') {
                        ret.value = col.substr(0, 30);
                        if (ret.value !== col) {
                            ret.valueFull = col;
                        }
                    } else if (col.length > 1000) {
                        ret.value = col.substr(0, 1000);
                        ret.valueFull = col;
                    }
                    ret.isGridInParamField = isGridInParamField;
                    return ret;
                }));
            });

            var pageSize = form.limit;
            var page = (form.offset / pageSize) + 1;

            var html = Mustache.render(template, {
                fields: fields,
                rows: rows,
                totalCount: res.totalCount,
                sql: res.sql,
                isStats: form.showStats,
                paging: self.makePaging(res, page, pageSize)
            });

            $result.html(html);

            document.querySelectorAll('pre code').forEach((block) => {
                hljs.highlightBlock(block);
            });

            $('.reqid-dropdown').on('show.bs.dropdown', function () {
                var reqid = $(this).data("reqid");
                let isTraceIdType = $(this).hasClass("reqid-type-traceId");
                var ulHtml = self.formatQuickLinks(form, reqid, isTraceIdType);
                $(this).find("ul").html(ulHtml);
            });

            $(".buttons").closest("td").hover(
                function() {
                    $(this).find(".buttons").show();
                },
                function() {
                    $(this).find(".buttons").hide();
                }
            );

            $result.find('.button-duplicate').on('click', function (e) {
                self.copyToClipboard(self.getFullValue(e.target));
            });

            $result.find('.button-add-to-filter').on('click', function (e) {
                var $td = $(e.target).closest('td');
                var val = self.getFullValue(e.target);

                var filter = val.replace(/[ ,]/g, "_");

                var field = $td.data('field');
                self.setFieldCheckbox(field, true);
                var $lf = self.addLogField(field);
                var inp = $lf.get(0);
                inp.value = inp.value ? inp.value + " " + filter : filter;
                $lf.focus();
            });

            $toggleParams = $('.toggle-params');
            $toggleParams.on('click', function (e) {
                var $td = $(e.target).closest('td');

                $td.find('.short-param').toggleClass('hidden');
                $td.find('.full-param').toggleClass('hidden');
            });

            if ($showParamsUncollapsed.prop('checked')) {
                $('.short-param.has-full').toggleClass('hidden');
                $('.full-param').toggleClass('hidden');
            }

            $('#sql_block').toggleClass('hidden', !$showSql.prop('checked'));
        },

        makePaging: function (res, page, pageSize) {
            var pageCount = res.totalCount === 0 ? 0 : Math.ceil(res.totalCount / pageSize);
            if (pageCount < 2) {
                return {pageSize: pageSize};
            }
            var showPageCount = Math.min(pageCount, 100);

            var pages = [];
            for (var i = 1; i <= showPageCount; i++) {
                pages.push({
                    page: i,
                    pageSize: pageSize,
                    isCurrent: i === page,
                });
            }

            return {
                pageSize: pageSize,
                morePages: pageCount > showPageCount,
                pages: pages,
            };
        },

        // конвертирует trace_id GrUT в trace_id Директа (см GrutTraceCallback)
        traceIdFromGrutToDirect: function(traceId) {
            let suffix = traceId.substring("d12ec700-0-".length).replaceAll("-", "");
            return hexToDec(suffix);
        },

        // конвертирует trace_id Директа в trace_id GrUT (см GrutTraceCallback)
        traceIdFromDirectToGrut: function(traceId) {
            let hex = decToHex(traceId, { prefix: false }).toLowerCase();
            let left = hex.substring(0, hex.length - 8);
            let right = hex.substring(hex.length - 8);
            return "d12ec700-0-" + left + "-" + right;
        },

        formatQuickLinks: function (form, reqid, isTraceId = false) {
            var html = [];
            let columnExtractor;
            if (isTraceId) {
                columnExtractor = (c) => c.traceIdColumn;
            } else {
                columnExtractor = (c) => c.reqidColumn;
            }
            let searchByReqId;
            // trace_id в messages_grut, начинающиеся с этого префикса, относятся к запросам в GrUT из Директа
            // из него можно вычислить trace_id запроса Директа, и попасть в соответствующие логи
            // поэтому если мы видим такой префикс, то декодируем его и используем для переходов в другие логи
            if (reqid.startsWith("d12ec700-0-")) {
                searchByReqId = self.traceIdFromGrutToDirect(reqid);
            } else {
                searchByReqId = reqid;
            }
            Object.keys(config.logs).map(function (it) {
                var lconf = config.logs[it];
                let idColumn = columnExtractor(lconf);
                if (idColumn !== undefined && idColumn !== "") {
                    var lcond = {};
                    if (it.startsWith("messages_grut")) {
                        let isNumeric = /^[0-9]+$/.test(searchByReqId);
                        if (isNumeric) {
                            // Если мы хотим перейти в логи GrUT'a по этому traceId,
                            // то мы должны преобразовать его по формуле
                            lcond[idColumn] = self.traceIdFromDirectToGrut(searchByReqId);
                        } else {
                            // Если searchByReqId не был сконвертирован в uint64-число (не начинался со спец префикса),
                            // то и обратная конвертация тоже не нужна
                            lcond[idColumn] = searchByReqId;
                        }
                    } else {
                        lcond[idColumn] = searchByReqId;
                    }
                    var lfields = lconf.columns.filter(function (x) {
                        return !x.hidden
                    }).map(function (x) {
                        return x.name
                    });
                    var lform = {
                        from: form.from,
                        to: form.to,
                        fields: lfields,
                        conditions: lcond,
                        limit: form.limit,
                        offset: 0,
                        showTraceIdRelated: false,
                        reverseOrder: false,
                    };
                    html.push("<li><a target=_blank href=\"#" + self.formatUrlHash(it, lform) + "\">" + it + "</a></li>");
                }
            });
            return html.join("\n");
        },

        formatUrlHash: function (logType, form) {
            return JSURL.stringify({
                logType: logType,
                form: {
                    ...form,
                    from: form.from.replace(/[:\-]/g, ''),
                    to: form.to.replace(/[:\-]/g, '')
                }
            }) + '$';
        },

        // ru.yandex.direct.logviewer.LogViewerApp -> r.y.d.logv.LogViewerApp
        shortClassName: function (className, len) {
            if (className === null) return null;
            className = "" + className;
            var parts = className.split(/\./);

            // остаток символов - для каждого подпакета как минимум точка и один символ и полное имя класса
            var rest = len - 2 * (parts.length - 1) - parts[parts.length - 1].length;
            var result = parts[parts.length - 1];

            for (var i = parts.length - 2; i >= 0; i--) {
                var s = parts[i].substr(0, Math.max(rest + 1, 1));
                rest -= s.length - 1;
                result = s + "." + result;
            }
            return result;
        },

        saveHistory: function (logType, form) {
            location.hash = self.formatUrlHash(logType, form);
        },

        loadHistory: function () {
            if (!location.hash) {
                return;
            }

            var hash = location.hash.substr(1);

            var isJson = hash.startsWith(encodeURIComponent("{"));
            var isJSURI = hash.startsWith("~");
            if (isJson || isJSURI) {
                if (isJSURI && hash.endsWith('$')) {
                    hash = hash.substring(0, hash.length - 1);
                }

                var state = (isJSURI ? JSURL.parse(hash) : JSON.parse(decodeURIComponent(hash)));

                var logType = state.logType;
                var form = state.form;

                $logTypeSelect.val(logType).change();

                var startDate;
                if (form.from === undefined) {
                    startDate = moment().startOf('day');
                } else if (form.from.startsWith('now')) {
                    // e.g. now-1h
                    const relTimePattern = /now(-\d+)([dhm])/g;
                    var match = relTimePattern.exec(form.from);
                    // default values
                    var num = -1;
                    var unit = 'hours';
                    if (match != null) {
                        num = +match[1];
                        switch (match[2]) {
                            case "d":
                                unit = 'days';
                                break;
                            case 'h':
                                unit = 'hours';
                                break;
                            case 'm':
                                unit = 'minutes';
                                break;
                        }
                    }
                    startDate = moment().add(num, unit);
                } else {
                    startDate = moment(form.from);
                }
                var endDate;
                if (form.to === undefined) {
                    endDate = moment().endOf('day');
                } else {
                    endDate = moment(form.to);
                }
                if (endDate.valueOf() < startDate.valueOf()) {
                    endDate = startDate.endOf('day');
                }

                var $logRange = $("#logRange");
                $logRange.data("daterangepicker").setStartDate(startDate);
                $logRange.data("daterangepicker").setEndDate(endDate);
                self.applyDateRange(startDate, endDate);

                if (form.showStats) {
                    $showStatsCheckbox.prop('checked', true);
                    $logTimeGroupBySelect.val(form.logTimeGroupBy).change();
                    $sortByCountCheckbox.prop('checked', form.sortByCount);
                }

                $traceidRelatedCheckbox.prop('checked', form.showTraceIdRelated);
                $reverseOrder.prop('checked', form.reverseOrder);

                var pageSize = form.limit;
                var page = (form.offset / pageSize) + 1;

                $pageSize.val(pageSize);
                if ($pageSize.val() == null) {
                    $("option", $pageSize).get(0).selected = true;
                }

                $("ul > li > a[data-field] span.glyphicon", $columnList).addClass("invisible");
                form.fields.forEach(function (field) {
                    $("ul > li > a[data-field=\"" + field + "\"] span.glyphicon", $columnList).removeClass("invisible");
                });

                self.setFieldConditions(form.conditions);

                self.showLogs(page, pageSize, "html");
            } else { // old format
                var params = hash.split(",");

                if (params[0] == 'dbName') {
                    params.shift();
                }

                $logTypeSelect.val(params.shift()).change();

                if (params.length === 0) {
                    return;
                }

                var startDate = moment(params.shift());
                var endDate = moment(params.shift());
                $("#logRange").data("daterangepicker").setStartDate(startDate);
                $("#logRange").data("daterangepicker").setEndDate(endDate);
                self.applyDateRange(startDate, endDate);

                if (params[0] == 'showStats') {
                    params.shift();
                    $showStatsCheckbox.prop("checked", true);
                    $logTimeGroupBySelect.val(params.shift()).change();
                    $sortByCountCheckbox.prop("checked", params.shift() === 'sortByCount');
                }

                var isBool = function (value) {
                    return value == "true" || value == "false";
                };

                if (isBool(params[0])) {
                    $traceidRelatedCheckbox.prop("checked", params.shift() == "true");
                }

                if (isBool(params[0])) {
                    $reverseOrder.prop("checked", params.shift() == "true");
                }

                var match = params.shift().match(/^(\d)+:(\d+)$/);
                var page = match[1];

                var pageSize = match[2];
                $pageSize.val(pageSize);
                if ($pageSize.val() == null) {
                    $("option", $pageSize).get(0).selected = true;
                }

                $("ul > li > a[data-field] span.glyphicon", $columnList).removeClass("invisible");
                var columnCount = params.shift();
                for (var i = 0; i < columnCount; i++) {
                    var col = params.shift();
                    $("ul > li > a[data-field=\"" + col + "\"] span.glyphicon", $columnList).addClass("invisible");
                }

                var conditions = {};
                for (var i = 0; i < params.length; i = i + 2) {
                    conditions[params[i]] = decodeURIComponent(params[i + 1]);
                }
                self.setFieldConditions(conditions);

                self.showLogs(page, pageSize, "html");
            }
        },

        copyToClipboard: function (data) {
            // https://hackernoon.com/copying-text-to-clipboard-with-javascript-df4d4988697f
            var el = document.createElement('textarea');
            el.value = data;
            el.setAttribute('readonly', '');
            el.style.position = 'absolute';
            el.style.left = '-9999px';

            document.body.appendChild(el);
            var selected = document.getSelection().rangeCount > 0
                ? document.getSelection().getRangeAt(0)
                : false;
            el.select();

            var copied = document.execCommand('copy');
            document.body.removeChild(el);
            if (selected) {
                document.getSelection().removeAllRanges();
                document.getSelection().addRange(selected);
            }
            return copied
        },

        getFullValue: function (clickTarget) {
            var $td = $(clickTarget).closest('td');
            var $short = $td.find('.short-param');
            var data = $short.text();
            if ($short.hasClass("has-full")) {
                data = $td.find('.full-param').text();
            }
            return data;
        }
    };
})();
