package ru.yandex.direct.oneshot;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.oneshot.app.AppInit;
import ru.yandex.direct.oneshot.configuration.AppConfiguration;
import ru.yandex.direct.oneshot.worker.ExecutionWorker;
import ru.yandex.direct.oneshot.worker.OneshotAliveChecker;
import ru.yandex.direct.oneshot.worker.ValidationWorker;
import ru.yandex.misc.sunMiscSignal.SunMiscSignal;

@ParametersAreNonnullByDefault
public class OneshotApp {

    private static final Logger logger = LoggingInitializer.getLogger(OneshotApp.class);

    public static void main(String[] args) {
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();
        OneshotParams oneshotParams = new OneshotParams();
        ParserWithHelp.parse(OneshotApp.class.getCanonicalName(), args, loggingParams, oneshotParams);
        LoggingInitializer.initialize(loggingParams, "direct.oneshot");

        AnnotationConfigApplicationContext appContext = createApplicationContext();

        if (!initApp(appContext)) {
            logger.error("Application can't do useful work, go to empty loop");
            emptyLoop();
            return;
        }

        Thread validationThread = new Thread(appContext.getBean(ValidationWorker.class),
                ValidationWorker.class.getSimpleName() + "Thread");
        validationThread.start();

        Thread executionThread = new Thread(appContext.getBean(ExecutionWorker.class),
                ExecutionWorker.class.getSimpleName() + "Thread");
        executionThread.start();

        Thread oneshotAliveCheckerThread = new Thread(appContext.getBean(OneshotAliveChecker.class),
                OneshotAliveChecker.class.getSimpleName() + "Thread");
        oneshotAliveCheckerThread.start();

        SunMiscSignal.Handler handler = signalName -> {
            logger.warn("Received a '{}' signal", signalName);

            if (appContext.isRunning()) {
                logger.warn("Starting graceful shutdown...");

                executionThread.interrupt();
                validationThread.interrupt();
                oneshotAliveCheckerThread.interrupt();

                try {
                    logger.warn("Waiting for workers termination");

                    validationThread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
                    executionThread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
                    oneshotAliveCheckerThread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN

                    logger.warn("Workers are terminated");
                } catch (InterruptedException ignored) {
                }
                appContext.close();
            } else {
                logger.error("Signal received twice, exiting...");
                System.exit(1);
            }
        };
        SunMiscSignal.handle("INT", handler);
        SunMiscSignal.handle("TERM", handler);
    }

    private static AnnotationConfigApplicationContext createApplicationContext() {
        AnnotationConfigApplicationContext appContext = new AnnotationConfigApplicationContext();
        appContext.register(AppConfiguration.class);
        appContext.registerShutdownHook();
        appContext.refresh();
        return appContext;
    }

    /**
     * @return true, если можно выполнять полезную работу, в противном случае false.
     */
    private static boolean initApp(ApplicationContext appContext) {
        AppInit appInit = appContext.getBean(AppInit.class);
        return appInit.init();
    }

    private static void emptyLoop() {
        while (!Thread.currentThread().isInterrupted()) {
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                return;
            }
        }
    }
}
