package ru.yandex.direct.oneshot.app;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.EnvironmentType;

import static ru.yandex.direct.dbschema.ppcdict.tables.Oneshots.ONESHOTS;

@Component
public class AppInit {

    private static final Logger logger = LoggerFactory.getLogger(AppInit.class);

    private OneshotVersionControl oneshotVersionControl;
    private EnvironmentType environmentType;
    private OneshotListUpdater oneshotListUpdater;
    private DslContextProvider dslContextProvider;

    public AppInit(OneshotVersionControl oneshotVersionControl, EnvironmentType environmentType,
                   OneshotListUpdater oneshotListUpdater, DslContextProvider dslContextProvider) {
        this.oneshotVersionControl = oneshotVersionControl;
        this.environmentType = environmentType;
        this.oneshotListUpdater = oneshotListUpdater;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * @return true, если данному инстансу можно делать полезную работу, и false в противном случае.
     */
    public boolean init() {
        return updateVersionAndOneshotListOnNewRelease();
    }

    /**
     * Проверка версии приложения и подготовка новой версии.
     *
     * Каждый запускающийся инстанс должен понять, является ли его версия актуальной или устаревшей.
     * Если версия актуальна и уже обновлена в базе, то инстанс просто переходит к выполнению полезной работы.
     * Если версия актуальна, но еще не обновлена в базе, то инстанс выставляет новую актуальную версию в базе
     * и переходит к выполнению полезной работы.
     * Если версия устаревшая, то есть до этого уже запускалась более свежая версия, то инстанс не должен
     * выполнять полезной работы, а вместо этого должен уйти в бесконечный цикл.
     *
     * @return true, если данному инстансу можно делать полезную работу, и false в противном случае.
     */
    private boolean updateVersionAndOneshotListOnNewRelease() {
        String myRevision = oneshotVersionControl.getCurrentVersion();
        if (oneshotVersionControl.isVersionUnknown(myRevision)) {
            if (environmentType.isDevelopment() || environmentType.isBeta()) {
                logger.warn("Can't find app version, running in wild west mode");
                oneshotListUpdater.initializeOneshotList(dslContextProvider.ppcdict());
                return true;
            } else {
                throw new RuntimeException("Can't get current application version");
            }
        }

        if (oneshotVersionControl.isRunningCurrentVersion()) {
            return true;
        }

        return dslContextProvider.ppcdict().transactionResult(conf -> {
            var dslContext = conf.dsl();
            // берём лок об записи в таблице oneshots.
            // выглядит странно, но т.к. в таблицу происходит запись только тут,
            // проблем быть не должно. По крайней мере на ТС и в проде.
            conf.dsl().select(ONESHOTS.ID).from(ONESHOTS).forUpdate();

            // если версия данного инстанса равна актуальной - выходим, можно делать полезную работу
            if (oneshotVersionControl.isRunningCurrentVersion()) {
                return true;
            }

            // если версия данного инстанса равна устаревшей - выходим, нельзя делать полезную работу
            if (oneshotVersionControl.isOldRevision(myRevision)) {
                return false;
            }

            // если версия данного инстанса не является текущей актуальной,
            // и при этом не числится среди последних запущенных,
            // то считаем, что данный инстанс - самый новый и должен
            // обновить в базе актуальную версию и список ваншотов
            oneshotListUpdater.initializeOneshotList(dslContext);

            oneshotVersionControl.recordNewRevision(dslContext, myRevision);
            return true;
        });
    }
}
