package ru.yandex.direct.oneshot.app;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.service.OneshotStartrekService;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.BaseOneshot;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.Retries;
import ru.yandex.direct.oneshot.worker.def.SafeOneshot;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;

import static java.util.function.Function.identity;
import static org.springframework.core.annotation.AnnotationUtils.findAnnotation;
import static org.springframework.core.annotation.AnnotationUtils.getAnnotation;
import static ru.yandex.direct.model.AppliedChanges.isChangedTo;
import static ru.yandex.direct.oneshot.worker.def.Retries.DEFAULT_TIMEOUT_SECONDS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class OneshotListUpdater {

    private static final Logger logger = LoggerFactory.getLogger(OneshotListUpdater.class);
    private final ApplicationContext appCtx;
    private final OneshotRepository oneshotRepository;
    private final OneshotStartrekService startrekService;

    public OneshotListUpdater(ApplicationContext appCtx, OneshotRepository oneshotRepository,
                              OneshotStartrekService startrekService) {
        this.appCtx = appCtx;
        this.oneshotRepository = oneshotRepository;
        this.startrekService = startrekService;
    }

    /**
     * Обновление списка ваншотов в базе.
     * <p>
     * Можно передать контекст открытой транзакции в PPCIDCT, чтобы изменения проводились в ней.
     * <p>
     * Если ваншот есть в базе, но его нет в приложении, тот помечается как удалённый.
     * <p>
     * Если ваншот есть и в базе и в приложении, у него могут быть обновлены поля
     * <ul>
     * <li>APPROVERS</li>
     * <li>MULTI_LAUNCH</li>
     * <li>SHARDED</li>
     * </ul>
     */
    public void initializeOneshotList(DSLContext dslContext) {
        Map<String, BaseOneshot> oneshotBeans = EntryStream.of(appCtx.getBeansOfType(BaseOneshot.class))
                .mapToKey((name, bean) -> bean.getClass().getCanonicalName())
                .toMap();
        List<Oneshot> currentOneshots = mapList(oneshotBeans.values(), OneshotListUpdater::beanToOneshot);

        Map<String, Oneshot> dbOneshots = StreamEx.of(oneshotRepository.getAll())
                .mapToEntry(Oneshot::getClassName, identity())
                .toMap();

        Predicate<Oneshot> existsInDb = oneshot -> dbOneshots.containsKey(oneshot.getClassName());

        List<Oneshot> newOneshots = StreamEx.of(currentOneshots)
                .remove(existsInDb)
                .peek(oneshot -> fillNewOneshotFields(oneshot))
                .toList();

        List<AppliedChanges<Oneshot>> currentOneshotsChanges = StreamEx.of(currentOneshots)
                .filter(existsInDb)
                .map(curOneshot -> {
                    var dbOneshot = dbOneshots.get(curOneshot.getClassName());
                    return applyChanges(dbOneshot, curOneshot);
                })
                .filter(AppliedChanges::hasActuallyChangedProps)
                .toList();

        List<AppliedChanges<Oneshot>> oneshotsDeletions = EntryStream.of(dbOneshots)
                .removeKeys(oneshotBeans::containsKey)
                .removeValues(Oneshot::getDeleted)
                .values()
                .map(dbOneshot -> ModelChanges.build(dbOneshot, Oneshot.DELETED, true).applyTo(dbOneshot))
                .toList();

        StreamEx.of(currentOneshotsChanges)
                .filter(isChangedTo(Oneshot.DELETED, false))
                .forEach(oneshotChanges -> startrekService.reopenTicket(oneshotChanges.getModel().getTicket()));
        oneshotsDeletions.forEach(oneshotChanges ->
                startrekService.closeTicket(oneshotChanges.getModel().getTicket()));

        List<AppliedChanges<Oneshot>> allOneshotsChanges = new ArrayList<>();
        allOneshotsChanges.addAll(currentOneshotsChanges);
        allOneshotsChanges.addAll(oneshotsDeletions);

        oneshotRepository.add(dslContext, newOneshots);
        oneshotRepository.update(dslContext, allOneshotsChanges);

        newOneshots.forEach(this::logNewOneshot);
        currentOneshotsChanges.forEach(this::logUpdatedOneshots);
        oneshotsDeletions.forEach(this::logDeletedOneshots);
    }

    private AppliedChanges<Oneshot> applyChanges(Oneshot dbOneshot, Oneshot curOneshot) {
        ModelChanges<Oneshot> modelChanges = new ModelChanges<>(dbOneshot.getId(), Oneshot.class)
                .process(curOneshot.getApprovers(), Oneshot.APPROVERS)
                .process(curOneshot.getRetries(), Oneshot.RETRIES)
                .process(curOneshot.getRetryTimeoutSeconds(), Oneshot.RETRY_TIMEOUT_SECONDS)
                .process(curOneshot.getPausedStatusOnFail(), Oneshot.PAUSED_STATUS_ON_FAIL)
                .process(curOneshot.getMultiLaunch(), Oneshot.MULTI_LAUNCH)
                .process(curOneshot.getSharded(), Oneshot.SHARDED)
                .process(curOneshot.getSafeOneshot(), Oneshot.SAFE_ONESHOT)
                .process(false, Oneshot.DELETED);

        return modelChanges.applyTo(dbOneshot);
    }

    private void logNewOneshot(Oneshot oneshot) {
        logger.info("New oneshot detected: {}, ticket {}", oneshot.getClassName(), oneshot.getTicket());
    }

    private void logUpdatedOneshots(AppliedChanges<Oneshot> appliedChanges) {
        logger.info("Oneshot {} has changed: {}", appliedChanges.getModel().getClassName(), appliedChanges);
    }

    private void logDeletedOneshots(AppliedChanges<Oneshot> appliedChanges) {
        logger.info("Oneshot {} was marked as deleted", appliedChanges.getModel().getClassName());
    }

    private void fillNewOneshotFields(Oneshot oneshot) {
        oneshot.setCreateTime(LocalDateTime.now());
        String ticketKey = startrekService.createTicket(oneshot);
        oneshot.setTicket(ticketKey);
    }

    /**
     * Читает класс бина-ваншота и составляет из него модель.
     */
    private static Oneshot beanToOneshot(BaseOneshot bean) {
        var beanClass = bean.getClass();
        var approvers = List.of(beanClass.getAnnotationsByType(Approvers.class)).stream()
                .flatMap(a -> List.of(a.value()).stream())
                .collect(Collectors.toSet());

        var isSafeOneshot = getAnnotation(beanClass, SafeOneshot.class) != null;

        var isMultilaunch = findAnnotation(beanClass, Multilaunch.class) != null;

        Retries retriesAnnotation = findAnnotation(beanClass, Retries.class);
        PausedStatusOnFail pausedStatusOnFailAnnotation = findAnnotation(beanClass, PausedStatusOnFail.class);
        var retries = retriesAnnotation != null ? retriesAnnotation.value() : 0;
        var retryTimeoutInSeconds = retriesAnnotation != null
                ? retriesAnnotation.timeoutSeconds() : DEFAULT_TIMEOUT_SECONDS;
        boolean pauseOnFail = pausedStatusOnFailAnnotation != null;

        var isSharded = ShardedOneshot.class.isAssignableFrom(beanClass);

        return new Oneshot()
                .withClassName(beanClass.getName())
                .withApprovers(approvers)
                .withSafeOneshot(isSafeOneshot)
                .withMultiLaunch(isMultilaunch)
                .withSharded(isSharded)
                .withPausedStatusOnFail(pauseOnFail)
                .withRetries(retries)
                .withRetryTimeoutSeconds(retryTimeoutInSeconds)
                .withDeleted(false);
    }
}
