package ru.yandex.direct.oneshot.app;

import java.util.ArrayList;
import java.util.Collections;

import org.jooq.DSLContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.version.DirectVersion;

import static ru.yandex.direct.common.db.PpcPropertyNames.ONESHOT_APP_CURRENT_REVISION;
import static ru.yandex.direct.common.db.PpcPropertyNames.ONESHOT_APP_LAST_REVISIONS;

@Component
public class OneshotVersionControl {
    private static final int LAST_REVISIONS_RECORD_COUNT = 10;

    private PpcPropertiesSupport ppcPropertiesSupport;

    public OneshotVersionControl(PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    /**
     * Каждый инстанс приложения при инициализации проверяет свежесть собственной версии
     * и при необходимости обновляет текущую актуальную версию в специальной ppc-property.
     * Здесь могут быть 3 варианта:
     * 1. версия данного инстанса самая новая и запускается впервые,
     * в этом случае инстанс при инициализации обновляет актуальную версию в специальной ppc-property,
     * а затем переходит к выполнению полезной работы;
     * 2. версия данного инстанса совпадает с актуальной версией в ppc-property, в этом случае
     * инстанс просто переходит к выполнению полезной работы.
     * 3. версия данного инстанса является устаревшей, в этом случае инстанс ничего не делает.
     * <p>
     * Если метод возвращает {@code true}, то это однозначно говорит о том, что версия данного инстанса
     * актуальна и инстанс может выполнять полезную работу. Но если метод возвращает {@code false},
     * то интерпретация зависит от этапа, на котором вызван метод: если на этапе инициализации,
     * то это может значить одно из двух, версия либо устаревшая, либо самая новая, запущенная впервые;
     * если после инициализации, то это значит, что версия инстанса является устаревшей (уже запущен
     * и проинициализирован более свежий инстанс).
     * <p>
     * Также возвращается {@code true} в случае, если версию не удалось получить.
     * По-хорошему это должно происходить только в разработческих средах.
     */
    public boolean isRunningCurrentVersion() {
        var myVersion = getCurrentVersion();
        // todo: наверно стоит убрать этот if, а вместо этого перестать использовать этот класс на разработческих средах
        if (isVersionUnknown(myVersion)) {
            return true;
        }

        var currentRevisionProp = ppcPropertiesSupport.get(ONESHOT_APP_CURRENT_REVISION);
        return myVersion.equals(currentRevisionProp.get());
    }

    /**
     * Возвращает ревизию из которой было собрано запущенное приложение.
     * Если ревизию получить не удалось, возвращает {@code DirectVersion.UNKNOWN_VERSION}
     */
    public String getCurrentVersion() {
        return DirectVersion.getVersion();
    }

    /**
     * @return {@code true}, если переданный параметр {@code currentVersion} это маркер неизвестной версии приложения
     */
    boolean isVersionUnknown(String currentVersion) {
        return DirectVersion.UNKNOWN_VERSION.equals(currentVersion);
    }

    /**
     * @return {@code true}, если указанная ревизия не является актуальной
     * запущенной ревизией и при этом является одной из последних {@value LAST_REVISIONS_RECORD_COUNT}
     * запущенных ревизий.
     */
    boolean isOldRevision(String someRevision) {
        var lastRevisions = ppcPropertiesSupport.get(ONESHOT_APP_LAST_REVISIONS).getOrDefault(Collections.emptyList());
        int someRevisionIndex = lastRevisions.indexOf(someRevision);
        return someRevisionIndex >= 0 && someRevisionIndex < lastRevisions.size() - 1;
    }

    /**
     * Записать указанную ревизию в {@code ppc_property} в качестве текущей, и в историю последних запущенных ревизий.
     */
    void recordNewRevision(DSLContext ppcdictDslCtx, String newRevision) {
        var lastRevisionsProp = ppcPropertiesSupport.get(ONESHOT_APP_LAST_REVISIONS);
        var lastRevisions = lastRevisionsProp.getOrDefault(Collections.emptyList());
        var newLastRevisions = new ArrayList<String>(LAST_REVISIONS_RECORD_COUNT + 1);
        newLastRevisions.addAll(lastRevisions);
        newLastRevisions.add(newRevision);
        if (newLastRevisions.size() > LAST_REVISIONS_RECORD_COUNT) {
            newLastRevisions.remove(0);
        }
        lastRevisionsProp.set(ppcdictDslCtx, newLastRevisions);
        ppcPropertiesSupport.get(ONESHOT_APP_CURRENT_REVISION).set(ppcdictDslCtx, newRevision);
    }
}
