package ru.yandex.direct.oneshot.debug;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.reflect.TypeToken;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.gson.Gson;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.oneshot.configuration.AppConfiguration;
import ru.yandex.direct.oneshot.util.GsonUtils;
import ru.yandex.direct.oneshot.worker.def.BaseOneshot;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;

import static ru.yandex.direct.oneshot.worker.ValidationWorker.validateOneshot;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectOneshotInputDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectShardedOneshotStateDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectSimpleOneshotStateDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.findOneshotBeanByClassName;

@ParametersAreNonnullByDefault
public class OneshotAppDebug {
    private static final Gson GSON = GsonUtils.getGSON();
    private static final Logger logger = LoggingInitializer.getLogger(OneshotAppDebug.class);
    private static final ThreadPoolExecutor EXECUTOR = new ThreadPoolExecutor(8, 8, 1, TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(),
            new ThreadFactoryBuilder()
                    .setDaemon(true)
                    .setNameFormat("debug-execution-thread-%d")
                    .build());

    public static void main(String[] args) {
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();
        OneshotParamsDebug oneshotParamsDebug = new OneshotParamsDebug();
        ParserWithHelp.parse(OneshotAppDebug.class.getCanonicalName(), args, loggingParams, oneshotParamsDebug);
        LoggingInitializer.initialize(loggingParams, "direct.oneshot");

        try (AnnotationConfigApplicationContext appContext = createApplicationContext()) {
            final BaseOneshot<?> oneshotBean = findOneshotBeanByClassName(appContext, oneshotParamsDebug.className);
            if (oneshotBean == null) {
                throw new RuntimeException(String.format("Can't find oneshot '%s'", oneshotParamsDebug.className));
            }

            if (!validateOneshot(appContext, oneshotParamsDebug.className, oneshotParamsDebug.inputData)) {
                throw new RuntimeException("Validation failed");
            }

            executeOneshot(appContext, oneshotBean, oneshotParamsDebug);
        }
    }

    private static <V, S> void executeOneshot(
            ApplicationContext appContext,
            BaseOneshot<V> baseOneshotBean,
            OneshotParamsDebug params
    ) {
        TypeToken<V> inputDataType = detectOneshotInputDataType(baseOneshotBean);
        V inputData = GSON.fromJson(params.inputData, inputDataType.getType());

        var retryTemplate = getRetryTemplate(params);

        S stateData;

        if (baseOneshotBean instanceof SimpleOneshot) {
            SimpleOneshot<V, S> simpleOneshotBean = (SimpleOneshot<V, S>) baseOneshotBean;
            TypeToken<S> stateDataType = detectSimpleOneshotStateDataType(simpleOneshotBean);
            stateData = GSON.fromJson(params.state, stateDataType.getType());

            do {
                var state = stateData;
                stateData = retryTemplate.execute(c -> simpleOneshotBean.execute(inputData, state));
                String stateStr = stateData != null ? GSON.toJson(stateData) : null;
                logger.info("oneshot state: {}", stateStr);
            } while (stateData != null);
        } else if (baseOneshotBean instanceof ShardedOneshot) {
            ShardedOneshot<V, S> shardedOneshotBean = (ShardedOneshot<V, S>) baseOneshotBean;
            TypeToken<S> stateDataType = detectShardedOneshotStateDataType(shardedOneshotBean);
            stateData = GSON.fromJson(params.state, stateDataType.getType());

            ShardHelper shardHelper = appContext.getBean(ShardHelper.class);
            final S finalStateData = stateData;
            shardHelper.forEachShardParallel(shard -> {
                if (!params.shards.isEmpty() && !params.shards.contains(shard)) {
                    return true;
                }
                S stateDataCurrent = finalStateData;
                do {
                    var state = stateDataCurrent;
                    stateDataCurrent = retryTemplate.execute(c ->
                            shardedOneshotBean.execute(inputData, state, shard)
                    );
                    logger.info("shard " + shard + ", state " + GSON.toJson(stateDataCurrent));
                } while (stateDataCurrent != null);
                return true;
            }, EXECUTOR);
        }
    }

    @NotNull
    private static RetryTemplate getRetryTemplate(OneshotParamsDebug params) {
        var backOffPolicy = new ExponentialBackOffPolicy();
        backOffPolicy.setMaxInterval(Duration.ofSeconds(30).toMillis());
        backOffPolicy.setInitialInterval(Duration.ofSeconds(5).toMillis());

        var retryPolicy = new SimpleRetryPolicy(params.retries + 1,
                Map.of(RuntimeException.class, true));

        var retryTemplate = new RetryTemplate();
        retryTemplate.setBackOffPolicy(backOffPolicy);
        retryTemplate.setRetryPolicy(retryPolicy);

        return retryTemplate;
    }

    private static AnnotationConfigApplicationContext createApplicationContext() {
        AnnotationConfigApplicationContext appContext = new AnnotationConfigApplicationContext();
        appContext.register(AppConfiguration.class);
        appContext.registerShutdownHook();
        appContext.refresh();
        return appContext;
    }
}
