package ru.yandex.direct.oneshot.oneshots.add_default_filter_condition

import org.jooq.DSLContext
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.model.StatusBLGenerated
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFiltersQueryFilter
import ru.yandex.direct.core.entity.performancefilter.model.Operator
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.FEEDS
import ru.yandex.direct.dbschema.ppc.enums.FeedsSource
import ru.yandex.direct.dbutil.QueryWithoutIndex
import ru.yandex.direct.dbutil.model.ClientId.fromLong
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.base.ShardedOneshotWithoutInput
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import java.time.LocalDateTime

/**
 * Ваншот для добавления условия available=true по умолчанию в фильтры смарт-групп по сайту
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "ali-al")
class AddDefaultPerformanceFilterConditionOneshot constructor(
    private val dslContextProvider: DslContextProvider,
    private val performanceFilterRepository: PerformanceFilterRepository,
    private val bannerCommonRepository: BannerCommonRepository,
    private val adGroupRepository: AdGroupRepository)
    : ShardedOneshotWithoutInput() {

    companion object {
        private val logger = LoggerFactory.getLogger(AddDefaultPerformanceFilterConditionOneshot::class.java)
    }

    override fun execute(shard: Int) {
        logger.info("Start on shard: $shard")

        val dslContext = dslContextProvider.ppc(shard)

        logger.info("Get filters to update")

        val clientIdByFeedId = getClientIdBySiteFeedId(dslContext)
        val feedIdByAdGroupId = getFeedIdByAdGroupId(dslContext, clientIdByFeedId.keys)

        val queryFilter = PerformanceFiltersQueryFilter.newBuilder()
            .withAdGroupIds(feedIdByAdGroupId.keys)
            .build()
        val filtersToChange = performanceFilterRepository.getFilters(shard, queryFilter)
            .filter { filter -> !filter.conditions.any {c -> c.fieldName == "available"} }

        logger.info("Add default site condition to ${filtersToChange.size} filters")

        filtersToChange
            .groupBy { clientIdByFeedId[feedIdByAdGroupId[it.pid]] }
            .forEach { (clientId, filters) ->
                logger.info("Update filters for clientId = $clientId with ids: ${filters.map { it.id }}")

                val localDateTime = LocalDateTime.now()

                val appliedChanges = filters
                    .map { filter ->
                        val newConditions = filter.conditions.toMutableList()
                        newConditions.add(PerformanceFilterCondition<Boolean>("available", Operator.EQUALS, "true").withParsedValue(true))
                        ModelChanges(filter.id, PerformanceFilter::class.java)
                            .process(newConditions, PerformanceFilter.CONDITIONS)
                            .process(localDateTime, PerformanceFilter.LAST_CHANGE)
                            .process(StatusBsSynced.NO, PerformanceFilter.STATUS_BS_SYNCED)
                            .applyTo(filter)
                    }

                val adGroupIds = filters.map { it.pid }.distinct()

                val adGroupById = adGroupRepository.getAdGroups(shard, adGroupIds).associateBy { it.id }

                val adGroupAppliedChanges = adGroupIds
                    .map { id ->
                        ModelChanges(id, PerformanceAdGroup::class.java)
                            .process(localDateTime, PerformanceAdGroup.LAST_CHANGE)
                            .process(StatusBLGenerated.PROCESSING, PerformanceAdGroup.STATUS_B_L_GENERATED)
                            .process(StatusBsSynced.NO, PerformanceAdGroup.STATUS_BS_SYNCED)
                            .applyTo(adGroupById[id] as PerformanceAdGroup)
                            .castModelUp(AdGroup::class.java)
                    }

                performanceFilterRepository.update(shard, appliedChanges)
                adGroupRepository.updateAdGroups(shard, fromLong(clientId!!), adGroupAppliedChanges)
                bannerCommonRepository.updateStatusBsSyncedByAdgroupId(shard, adGroupIds, StatusBsSynced.NO)
            }

        logger.info("Finish on shard: $shard")
    }

    @QueryWithoutIndex("На каждом шарде до 500 записей")
    private fun getClientIdBySiteFeedId(dslContext: DSLContext): Map<Long, Long> {
        return dslContext
            .select(FEEDS.FEED_ID, FEEDS.CLIENT_ID)
            .from(FEEDS)
            .where(FEEDS.SOURCE.eq(FeedsSource.site))
            .fetchMap(FEEDS.FEED_ID, FEEDS.CLIENT_ID)
    }

    private fun getFeedIdByAdGroupId(dslContext: DSLContext, feedIds: Set<Long>): Map<Long, Long> {
        return dslContext
            .select(ADGROUPS_PERFORMANCE.PID, ADGROUPS_PERFORMANCE.FEED_ID)
            .from(ADGROUPS_PERFORMANCE)
            .where(ADGROUPS_PERFORMANCE.FEED_ID.`in`(feedIds))
            .fetchMap(ADGROUPS_PERFORMANCE.PID, ADGROUPS_PERFORMANCE.FEED_ID)
    }
}
