package ru.yandex.direct.oneshot.oneshots.add_default_filter_condition

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils.getHashForDynamicFeedRules
import ru.yandex.direct.core.entity.feed.model.Source
import ru.yandex.direct.core.entity.performancefilter.model.Operator
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject

data class InputData(
    val campaignIds: List<Long>
)

/**
 * Ваншот для добавления условия available!=false по умолчанию в фильтры ТК
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "ali-al")
class AddDefaultUacFilterConditionOneshot constructor(
    private val shardHelper: ShardHelper,
    private val campaignRepository: CampaignRepository,
    private val adGroupRepository: AdGroupRepository,
    private val performanceFilterRepository: PerformanceFilterRepository,
    private val dynamicTextAdTargetRepository: DynamicTextAdTargetRepository,
    private val performanceFilterService: PerformanceFilterService,
    private val dynamicTextAdTargetService: DynamicTextAdTargetService
): SimpleOneshot<InputData, Void> {
    companion object {
        private const val AVAILABLE = "available"
        private const val VALUE_FALSE = "false"
        private const val PERFORMANCE = "performance"
        private const val DYNAMIC = "dynamic"
        private const val NO_FILTERS = "No %s filters to update"
        private val logger = LoggerFactory.getLogger(AddDefaultUacFilterConditionOneshot::class.java)
    }

    override fun validate(inputData: InputData) = validateObject(inputData) {
        listProperty(inputData::campaignIds)
            .check(notEmptyCollection())
            .checkEach(greaterThan(0L))
    }

    override fun execute(inputData: InputData, prevState: Void?): Void? {
        val shardToCids = inputData.campaignIds.groupBy { shardHelper.getShardByCampaignId(it) }
        shardToCids.forEach { addDefaultUacFilterCondition(it.key, it.value) }

        return null
    }

    private fun addDefaultUacFilterCondition(shard: Int, cids: List<Long>) {
        val ecomCampaigns = campaignRepository.getCampaigns(shard, cids)
            .filter { it.metatype == CampaignMetatype.ECOM && !it.statusArchived}
        logger.info("Updating filters for ${ecomCampaigns.size} ecom campaigns")

        val adgroupIdsByCid = adGroupRepository.getAdGroupIdsByCampaignIds(shard, ecomCampaigns.map {it.id})

        ecomCampaigns
            .groupBy { ClientId.fromLong(it.clientId) }
            .forEach { (clientId, campaigns) ->
                val pids = campaigns.flatMap { adgroupIdsByCid.getOrDefault(it.id, emptyList()) }

                val performanceFilters = performanceFilterRepository.getFiltersByAdGroupIds(shard, pids)
                    .flatMap { it.value }
                    .filter { it.isDeleted == false }
                    .filter { !it.conditions.any { it.fieldName == AVAILABLE && it.operator == Operator.NOT_EQUALS } }
                    .map {
                        val newConditions = if (it.source == Source.SITE) {
                            it.conditions.filter { it.fieldName != AVAILABLE }.toMutableList()
                        } else {
                            it.conditions.toMutableList()
                        }
                        newConditions.add(
                            PerformanceFilterCondition<Boolean>(AVAILABLE, Operator.NOT_EQUALS, VALUE_FALSE)
                                .withParsedValue(false)
                        )
                        it.withConditions(newConditions)
                    }

                if (performanceFilters.isEmpty()) {
                    logger.info(NO_FILTERS.format(PERFORMANCE))
                } else {
                    val result = performanceFilterService
                        .updatePerformanceFilters(clientId, campaigns[0].userId, performanceFilters)

                    val updateInfoByIsSuccess = performanceFilters.asSequence()
                        .map { it.id }
                        .zip(result.result.asSequence())
                        .partition { it.second.result != null }

                    logResult(PERFORMANCE, updateInfoByIsSuccess)
                }

                val dynamicFeedAdTargets = dynamicTextAdTargetRepository
                    .getDynamicFeedAdTargetsByAdGroupIds(shard, clientId, pids)
                    .filter { !it.condition.any { it.fieldName == AVAILABLE } }

                val dynamicModelChanges = dynamicFeedAdTargets
                    .map {
                        val newConditions = it.condition.toMutableList()
                        newConditions.add(
                            DynamicFeedRule<Boolean>(AVAILABLE, Operator.NOT_EQUALS, VALUE_FALSE)
                                .withParsedValue(false)
                        )
                        ModelChanges(it.id, DynamicFeedAdTarget::class.java)
                            .processNotNull(newConditions, DynamicFeedAdTarget.CONDITION)
                            .processNotNull(getHashForDynamicFeedRules(newConditions), DynamicFeedAdTarget.CONDITION_HASH)
                    }

                if (performanceFilters.isEmpty()) {
                    logger.info(NO_FILTERS.format(DYNAMIC))
                } else {
                    val result = dynamicTextAdTargetService
                        .updateDynamicFeedAdTargets(clientId, campaigns[0].userId, dynamicModelChanges)

                    val updateInfoByIsSuccess = dynamicFeedAdTargets.asSequence()
                        .map { it.id }
                        .zip(result.result.asSequence())
                        .partition { it.second.result != null }

                    logResult(DYNAMIC, updateInfoByIsSuccess)
                }
            }
    }

    private fun logResult(filterType: String, updateInfoByIsSuccess: Pair<List<Pair<Long, Result<Long>>>, List<Pair<Long, Result<Long>>>>) {
        if (updateInfoByIsSuccess.first.isNotEmpty()) {
            logger.info("Updated $filterType filters with ids: ${updateInfoByIsSuccess.first.map { it.first }}")
        }

        if (updateInfoByIsSuccess.second.isNotEmpty()) {
            logger.error("Failed to update $filterType filters: ${updateInfoByIsSuccess.second
                .map { "filter id = ${it.first} with errors = ${it.second.errors }" }}")
        }
    }
}
