package ru.yandex.direct.oneshot.oneshots.add_performance_main_banners

import java.io.ByteArrayOutputStream
import java.util.concurrent.TimeUnit
import one.util.streamex.StreamEx
import org.asynchttpclient.AsyncCompletionHandler
import org.asynchttpclient.AsyncHandler
import org.asynchttpclient.AsyncHttpClient
import org.asynchttpclient.HttpResponseBodyPart
import org.asynchttpclient.Response
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.mock.web.MockMultipartFile
import org.springframework.stereotype.Component
import ru.yandex.direct.bannerstorage.client.BannerStorageClient
import ru.yandex.direct.bannerstorage.client.Utils
import ru.yandex.direct.bannerstorage.client.model.Creative
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService
import ru.yandex.direct.core.entity.image.container.BannerImageType
import ru.yandex.direct.core.entity.image.model.ImageUploadContainer
import ru.yandex.direct.core.entity.image.service.ImageService
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE_MAIN
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.isNull
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.CommonConstraints.notTrue
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class InputData(
    val operatorUid: Long?,
    val ytCluster: YtCluster,
    val tablePath: String,
    val enableCreativeFreeInterface: Boolean?, // включает CREATIVE_FREE_INTERFACE клиенту после его полной обработки
    val campaignIdsOnly: List<Long>?,  // можно дополнительно указать жёсткий фильтр по cid
    val partial: Boolean? // разрешает частичную обработку (иначе обработаются либо все заказанные кампании клиента, либо никакие)
)

data class State(
    val lastRow: Long
)

class InputTableRow : YtTableRow(listOf(CLIENT_ID)) {
    companion object {
        private val CLIENT_ID = YtField("ClientID", Long::class.java)
    }

    val clientId: Long?
        get() = valueOf(CLIENT_ID)
}

/**
 * Ваншот для генерации и добавления смартбаннера нового типа (performance_main)
 * Принимает на вход список клиентов, ищет у них неархивные performance-кампании и непустые группы в них
 * Если в найденных группах нет performance_main-баннера, то добавляет туда, если позволяет валидация
 * Если валидация не позволяет, то пишет в лог ошибку и пропускает эту группу
 */
@Component
@Multilaunch
@Approvers("elwood", "mspirit", "buhter", "ulyashevda", "poldnev")
@Retries(5)
@PausedStatusOnFail
class AddPerformanceMainBannersOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
    private val bannersAddOperationFactory: BannersAddOperationFactory,
    private val bannerStorageClient: BannerStorageClient,
    private val imageService: ImageService,
    private val asyncHttpClient: AsyncHttpClient,
    private val featureManagingService: FeatureManagingService
) : SimpleOneshot<InputData, State?> {
    companion object {
        private val logger = LoggerFactory.getLogger(AddPerformanceMainBannersOneshot::class.java)
        private const val CHUNK_SIZE = 100

        // супер-пользователь, от имени которого будут выполняться операции добавления баннера
        // TODO: посмотреть на ТС, прорастают ли эти события в пользологи
        private const val SUPER_OPERATOR_UID = 933325662L
    }

    override fun validate(inputData: InputData) = validateObject(inputData) {
        property(inputData::operatorUid) {
            check(validId(), When.notNull())
        }
        property(inputData::ytCluster) {
            check(notNull())
        }
        property(inputData::tablePath) {
            check(notNull())
            check(
                fromPredicate(
                    { checkIfInputTableExists(inputData.ytCluster, it) },
                    CommonDefects.objectNotFound()
                )
            )
        }
        property(inputData::campaignIdsOnly) {
            check(isNull(), When.isTrue(inputData.enableCreativeFreeInterface ?: false))
        }
        property(inputData::partial) {
            check(notTrue(), When.isTrue(inputData.enableCreativeFreeInterface ?: false))
        }
    }

    override fun execute(inputData: InputData, prevState: State?): State? {
        logger.info("Start from state=$prevState")
        val operatorUid = inputData.operatorUid ?: SUPER_OPERATOR_UID
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + CHUNK_SIZE
        val clientIdsChunk = readClientIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        val groupByShard = shardHelper.groupByShard(clientIdsChunk, ShardKey.CLIENT_ID)
        groupByShard.forEach { shard, clientIds ->
            logger.info("Processing shard $shard")
            var adgroups = getAdGroupsToProcess(shard, clientIds)
            // если указаны конкретные cid'ы, то всё остальное скипаем
            if (inputData.campaignIdsOnly != null) {
                adgroups = adgroups.filter { inputData.campaignIdsOnly.contains(it.cid) }
            }
            val adGroupsByClientId = adgroups.groupBy { ClientId.fromLong(it.clientId) }
            logger.info("Got ${adgroups.size} adGroups among ${adGroupsByClientId.size} clients")
            adGroupsByClientId.forEach { (clientId, clientAdGroups) ->
                logger.info("Processing client $clientId with ${clientAdGroups.size} adGroups")
                if (inputData.partial == true) {
                    for (adGroup in clientAdGroups) {
                        processAdGroups(shard, clientId, operatorUid, listOf(adGroup))
                    }
                } else {
                    val isSuccessful = processAdGroups(shard, clientId, operatorUid, clientAdGroups)
                    if (isSuccessful && inputData.enableCreativeFreeInterface == true) {
                        logger.info("Enabling creative free interface for client $clientId")
                        featureManagingService.enableFeatureForClient(clientId, FeatureName.CREATIVE_FREE_INTERFACE)
                    }
                }
            }
        }
        return if (clientIdsChunk.size == CHUNK_SIZE) State(lastRow) else null
    }

    private fun processAdGroups(shard: Int, clientId: ClientId, operatorUid: Long, adGroups: List<AdGroupToProcess>): Boolean {
        logger.info("Processing client $clientId: adGroups $adGroups")

        try {
            val banners = adGroups.map {
                val logoImageHash = uploadLogoFromBannerStorageToDirect(shard, clientId, it.cid, it.pid)

                PerformanceBannerMain()
                    .withAdGroupId(it.pid)
                    .withLogoImageHash(logoImageHash)
            }

            val result = bannersAddOperationFactory.createFullAddOperation(banners, clientId, operatorUid, false)
                .prepareAndApply()

            val successResults = getSuccessResults(result)
            if (successResults.isNotEmpty()) {
                logger.info("Added bannerIds: $successResults")
            }
            if (result.validationResult != null && result.validationResult.hasAnyErrors()) {
                logger.error("Banners validationErrors: ${result.validationResult.flattenErrors()}")
            }

            return result.isSuccessful
        } catch (e: Exception) {
            logger.error("Unexpected exception when processing adGroups $adGroups", e)
            return false
        }
    }

    /**
     * Функция пытается получить логотип с креативов на группе и загрузить его в Директ как ASSET_LOGO
     * При успешном результате возвращает imageHash загруженной (или уже имеющейся в базе) картинки, иначе null
     */
    private fun uploadLogoFromBannerStorageToDirect(shard: Int, clientId: ClientId, cid: Long, pid: Long): String? {
        val creativeIds = getCreativeIds(shard, cid, pid)
        if (creativeIds.isEmpty()) {
            return null
        }

        val creatives = bannerStorageClient.getCreatives(creativeIds.map { it.toInt() })

        // Метод при получении логотипа игнорирует неактуальные макеты
        // Если актуальных макетов в группе вообще нет, то берется логотип с неактуальных макетов
        // Такая логика полезна, так как некоторое время назад отключили редактирование креативов устаревших макетов
        // и при замене логотипа он менялся только у креативов с актуальными макетами (null считаем валидным логотипом)
        // Таким образом, логотипы на актуальных макетах стали более приоритетными, однако в базе есть группы,
        // в которых вообще нет креативов на новых лейаутах, и по таким группам тоже надо уметь получать логотип
        val logoFileId = getLogoParameterValue(creatives) ?: return null

        val logoFile = bannerStorageClient.getFile(logoFileId)

        val bytes = downloadFile(logoFile.stillageFileUrl)
        val massResult = imageService.saveImages(
            clientId,
            mapOf(
                0 to ImageUploadContainer(
                    0,
                    MockMultipartFile(logoFile.fileName, bytes),
                    null
                )
            ),
            BannerImageType.ASSET_LOGO_EXTENDED
        )

        val successResults = getSuccessResults(massResult)
        if (successResults.isNotEmpty()) {
            val successResult = successResults[0]
            logger.info("Uploaded image: ${successResult.imageHash}")
            return successResult.imageHash
        }
        if (massResult.validationResult != null && massResult.validationResult.hasAnyErrors()) {
            logger.error("LogoImage validationErrors: ${massResult.validationResult.flattenErrors()}")
        } else {
            logger.error("Neither validationErrors nor successResult received")
        }
        return null
    }

    /**
     * Извлекает из группы креативов значение логотипа, если оно одинаковое у всех креативов
     * Если параметр где-то отсутствует или имеет отличное от других значение
     * (если креатив был отредактирован отдельно), то функция возвращает то значение, которое встречается чаще других
     * Игнорирует креативы с устаревшими макетами, если есть хотя бы один с не устаревшим макетом
     * Отсутствие логотипа считается валидным подвидом логотипа (вариант "без логотипа" в интерфейсе)
     */
    private fun getLogoParameterValue(creatives: List<Creative>): Int? {
        val allVariants: List<Int?> = getLogoParameterValues(creatives, useObsoleteLayouts = false)
            .takeIf { it.isNotEmpty() } ?: getLogoParameterValues(creatives, useObsoleteLayouts = true)
        return allVariants.groupBy { it }
            .mapValues { it.value.size }
            .entries.maxByOrNull { it.value }
            ?.key
    }

    private fun getLogoParameterValues(creatives: List<Creative>, useObsoleteLayouts: Boolean): List<Int?> {
        return creatives.asSequence()
            .filter { useObsoleteLayouts || !Utils.isPerformanceLayoutObsolete(it.layoutCode?.layoutId) }
            .map { it.parameters.find { it.paramName == "LOGO" } }
            .map { it?.values?.getOrNull(0)?.toInt() }
            .toList()
    }

    private fun downloadFile(url: String): ByteArray {
        val stream = ByteArrayOutputStream()
        val future = asyncHttpClient.prepareGet(url).execute(object : AsyncCompletionHandler<ByteArrayOutputStream>() {
            override fun onBodyPartReceived(bodyPart: HttpResponseBodyPart): AsyncHandler.State {
                stream.write(bodyPart.bodyPartBytes)
                return AsyncHandler.State.CONTINUE
            }

            override fun onCompleted(response: Response?): ByteArrayOutputStream {
                return stream
            }
        })
        future.get(30, TimeUnit.SECONDS)
        return stream.toByteArray()
    }

    private fun <T> getSuccessResults(massResult: MassResult<T>): List<T> {
        return if (massResult.result == null) {
            emptyList()
        } else StreamEx.of(massResult.result)
            .nonNull()
            .filter { it.isSuccessful }
            .map { it.result }
            .nonNull()
            .toList()
    }

    data class AdGroupToProcess(
        val clientId: Long,
        val cid: Long,
        val pid: Long
    )

    private fun getCreativeIds(shard: Int, cid: Long, pid: Long): MutableList<Long> {
        return dslContextProvider.ppc(shard)
            .select(BANNERS_PERFORMANCE.CREATIVE_ID)
            .from(BANNERS_PERFORMANCE)
            .where(BANNERS_PERFORMANCE.CID.eq(cid).and(BANNERS_PERFORMANCE.PID.eq(pid)))
            .fetch { it.get(BANNERS_PERFORMANCE.CREATIVE_ID) }
    }

    private fun getAdGroupsToProcess(shard: Int, clientIds: List<Long>): List<AdGroupToProcess> {
        return dslContextProvider.ppc(shard)
            .select(
                CAMPAIGNS.CLIENT_ID,
                CAMPAIGNS.CID,
                PHRASES.PID
            )
            .from(CAMPAIGNS)
            .join(PHRASES).on(CAMPAIGNS.CID.eq(PHRASES.CID))
            .leftJoin(BANNERS_PERFORMANCE_MAIN).on(BANNERS_PERFORMANCE_MAIN.PID.eq(PHRASES.PID))
            .join(BANNERS).on(
                BANNERS.CID.eq(CAMPAIGNS.CID)
                    .and(BANNERS.PID.eq(PHRASES.PID))
            )
            .where(
                CAMPAIGNS.CLIENT_ID.`in`(clientIds)
                    .and(CAMPAIGNS.TYPE.eq(CampaignsType.performance))
                    .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                    .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))  // empty-кампании не видны для operator'a
                    .and(BANNERS_PERFORMANCE_MAIN.PID.isNull)
            )
            .groupBy(
                CAMPAIGNS.CLIENT_ID,
                CAMPAIGNS.CID,
                PHRASES.PID
            )
            // Не добавляем в пустые группы: в них новый баннер автоматически добавится при редактировании группы
            .having(DSL.count().greaterThan(0))
            .fetch { AdGroupToProcess(it.get(CAMPAIGNS.CLIENT_ID), it.get(CAMPAIGNS.CID), it.get(PHRASES.PID)) }
    }

    private fun checkIfInputTableExists(ytCluster: YtCluster, tablePath: String): Boolean {
        return ytProvider.getOperator(ytCluster).exists(YtTable(tablePath))
    }

    private fun readClientIdsFromYtTable(
        ytCluster: YtCluster,
        tablePath: String,
        startRow: Long,
        lastRow: Long
    ): List<Long> {
        val clientIds = mutableListOf<Long>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(
                YtTable(tablePath),
                { clientIds.add(it.clientId!!) }, InputTableRow(), startRow, lastRow
            )
        return clientIds
    }
}
