package ru.yandex.direct.oneshot.oneshots.addspravcounter;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.repository.CampMetrikaCountersRepository;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.oneshot.oneshots.addspravcounter.service.SpravService;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Данный ваншот добавляет на кампанию счётчики метрики, которых на ней нет, но есть на организации одного из баннеров
 * этой кампании.
 *
 * Алгоритм следущий:
 * - вытаскиваем для баннеров manual пермалинки
 * - достаём из YT привязку id пермалинка - id счётчика метрики
 * - для кампаний строим обязательные счётчики метрики, которые должны быть на кампании, поскольку они связаны с
 *   пермалинком баннера этой кампании
 * - построенные обязательные счётчики добавляем к текущим счётчикам кампании и получаем счётчики кампании, которые
 *   нужно записать в базу, и записываем их.
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers({"ssdmitriev", "buhter", "pavryabov"})
public class AddSpravCounterOneshot implements ShardedOneshot<Void, AddSpravState> {
    private static final Logger logger = LoggerFactory.getLogger(AddSpravCounterOneshot.class);

    private static final int LIMIT = 5000;

    private final SpravService spravService;
    private final OrganizationRepository organizationRepository;
    private final CampMetrikaCountersRepository campMetrikaCountersRepository;
    private final BannerRelationsRepository bannerRelationsRepository;

    @Autowired
    public AddSpravCounterOneshot(SpravService spravService, OrganizationRepository organizationRepository,
                                  CampMetrikaCountersRepository campMetrikaCountersRepository,
                                  BannerRelationsRepository bannerRelationsRepository) {
        this.spravService = spravService;
        this.organizationRepository = organizationRepository;
        this.campMetrikaCountersRepository = campMetrikaCountersRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
    }

    @Nullable
    @Override
    public AddSpravState execute(Void inputData, AddSpravState prevState, int shard) {
        //если предыдущая стадия не инициализирована -- начали ваншот
        if (prevState == null) {
            logger.info("First iteration! shard: {}", shard);
            return new AddSpravState(0, 0);
        }

        Map<Long, Long> permalinkByBannerId = organizationRepository.getPermalinkIdByBannerId(shard, LIMIT,
                prevState.getLastPermalinkId(), prevState.getLastBannerId());

        //если пытаемся вытащить еще пермалинки баннеров, а их больше нет, можно заканчивать
        if (permalinkByBannerId.isEmpty()) {
            logger.info("Last iteration, last permalink id: {}, last banner id: {}, shard: {}",
                    prevState.getLastPermalinkId(), prevState.getLastBannerId(), shard);
            return null;
        }

        logger.info("Last permalink id: {}, last banner id: {}, shard: {}", prevState.getLastPermalinkId(),
                prevState.getLastBannerId(), shard);

        //получаем из yt данные о привязанном к пермалинку счетчике
        Map<Long, Long> counterIdByPermalinkId =
                spravService.getCounterIdByPermalinkId(Set.copyOf(permalinkByBannerId.values()));

        //получаем для баннера id кампании
        Map<Long, Long> campaignIdsByBannerIds = bannerRelationsRepository.getCampaignIdsByBannerIdsForShard(shard,
                permalinkByBannerId.keySet());

        Set<Long> uniqueCampaignIds = Set.copyOf(campaignIdsByBannerIds.values());

        //для кампаний получаем счетчики метрики
        var actualMetrikaCountersByCampaignId = campMetrikaCountersRepository
                .getMetrikaCounterByCid(shard, uniqueCampaignIds);
        var counterById = StreamEx.of(actualMetrikaCountersByCampaignId.values())
                .nonNull()
                .flatMap(Collection::stream)
                .distinct(MetrikaCounter::getId)
                .mapToEntry(MetrikaCounter::getId, Function.identity())
                .toMap();

        //для кампаний вычисляем id счетчиков метрики
        var actualMetrikaCounterIdsByCampaignId = EntryStream.of(actualMetrikaCountersByCampaignId)
                .mapValues(counters -> listToSet(counters,
                        MetrikaCounter::getId))
                .toMap();

        //для кампаний вычисляем новые id счетчиков
        Map<Long, Set<Long>> permalinkCounterIdsByCampaignId = EntryStream.of(campaignIdsByBannerIds)
                .invert()
                .mapValues(permalinkByBannerId::get)
                .nonNullValues()
                .mapValues(counterIdByPermalinkId::get)
                .nonNullValues()
                .grouping(Collectors.toSet());

        Map<Long, Set<Long>> newMetrikaCounterIdsByCampaignId = StreamEx.of(uniqueCampaignIds)
                .mapToEntry(cid -> StreamEx.of(actualMetrikaCounterIdsByCampaignId.getOrDefault(cid, emptySet()))
                        .append(permalinkCounterIdsByCampaignId.getOrDefault(cid, emptySet()))
                        .toSet())
                .toMap();

        //в метрику ходить плохо для вычисления ecommerсe поэтому берем значения из уже лежавших в базе,
        //а для счетчиков пермлаников выставляем флаг в false
        Map<Long, List<MetrikaCounter>> countersByCampaignId = EntryStream.of(newMetrikaCounterIdsByCampaignId)
                .mapToValue((campaignId, counterIds) -> calculateCounterForCounterIds(counterById, counterIds,
                        permalinkCounterIdsByCampaignId.get(campaignId)))
                .removeValues(List::isEmpty)
                .toMap();

        //в рамках обновления -- добавляются записи в две таблицы camp_metrika_counters и metrika_counters
        campMetrikaCountersRepository.updateMetrikaCounters(shard, countersByCampaignId);

        return new AddSpravState(Collections.max(permalinkByBannerId.values()),
                Collections.max(permalinkByBannerId.keySet()));
    }

    private List<MetrikaCounter> calculateCounterForCounterIds(
            Map<Long, MetrikaCounter> counterById,
            Set<Long> counterIds,
            @Nullable Set<Long> permalinkCounterIds) {
        return StreamEx.of(counterIds)
                .map(counterId -> counterById.getOrDefault(counterId,
                        new MetrikaCounter().withId(counterId)))
                .map(counter -> new MetrikaCounter()
                        .withId(counter.getId())
                        .withSource(getSource(counter, permalinkCounterIds))
                        .withHasEcommerce(counter.getHasEcommerce()))
                .toList();
    }

    private MetrikaCounterSource getSource(MetrikaCounter counter,
                                           @Nullable Set<Long> permalinkCounterIds) {
        if (permalinkCounterIds != null && permalinkCounterIds.contains(counter.getId())) {
            return MetrikaCounterSource.SPRAV;
        }

        if (counter.getSource() == null) {
            return MetrikaCounterSource.UNKNOWN;
        }

        return counter.getSource();
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }
}
