package ru.yandex.direct.oneshot.oneshots.archive_campaigns_for_recomendation

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.utils.CommonUtils.nvl
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject

private const val CLIENTS_FOR_ITERATION_NUMBER_LIMIT = 5000

@Component
@Multilaunch
@Approvers("ali-al", "buhter", "munira")
class CampaignsForRecommendationArchiveOneshot(
    private val shardHelper: ShardHelper,
    private val rbacService: RbacService,
    private val campaignRepository: CampaignRepository,
    private val campaignsService: CampaignService,
) : ShardedOneshot<InputData, Void> {
    override fun validate(inputData: InputData) = validateObject(inputData) {
        listProperty(inputData::clientIds)
            .check(notNull())
            .check(CollectionConstraints.maxListSize(CLIENTS_FOR_ITERATION_NUMBER_LIMIT))
            .checkEach(CommonConstraints.validId(), When.isValid())
            .checkEach(
                Constraint.fromPredicate(
                    { clientId -> shardHelper.isExistentClientId(clientId!!) }, CommonDefects.objectNotFound()
                ), When.isValid()
            )
    }

    override fun execute(inputData: InputData, prevState: Void?, shard: Int): Void? {
        val clientIds = inputData.clientIds
        val campaignIds = campaignRepository.getCampaignIdsForStopping(shard, clientIds)
        campaignRepository.stopCampaigns(shard, campaignIds.values.flatten())

        sleep()

        clientIds.forEach { id ->
            val clientId = ClientId.fromLong(id)
            val operatorUid = rbacService.getChiefByClientId(clientId)
            campaignsService.archiveCampaigns(nvl(campaignIds[id], emptyList()), operatorUid, clientId)
        }

        return null
    }

    private fun sleep() {
        try {
            Thread.sleep(1000 * 60 * 65)
        } catch (e: InterruptedException) {
            throw RuntimeException(e)
        }
    }
}

data class InputData(
    val clientIds: List<Long>
)
